<?php

namespace App\Integrations;

use App\Models\SalesChannelType;
use App\SDKs\WalmartSDK\Enum;
use App\SDKs\WalmartSDK\WalmartAPI;
use Carbon\Carbon;
use Generator;

class Walmart extends Channel implements HasOAuth
{
    private $walmartSDK;

    public function __construct($credentials)
    {
        if (! empty($credentials)) {
            if (! array_key_exists('marketplace', $credentials)) {
                throw new \InvalidArgumentException('marketplace is required');
            }

            if (! array_key_exists('is_oauth', $credentials)) {
                throw new \InvalidArgumentException('is_oauth is required');
            }

            $this->walmartSDK = new WalmartAPI($credentials, $credentials['marketplace'], boolval($credentials['is_oauth']));
        }
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        /**
         * Test request on Orders with limit 1.
         */
        $params = [
            'createdStartDate' => date('Y-m-d', time() - 5184000),
            'limit' => 1,
        ];

        $test_response = $this->walmartSDK->getOrders($params);
        if (isset($test_response['error'])) {
            return ['status' => false, 'message' => $test_response['error']];
        }

        return ['status' => true, 'response' => []];
    }

    /**
     * Generate OAuth user url.
     */
    public function generateOAuthUrl(array $options = []): array
    {
        if (empty($options['developer_id'])) {
            throw new \InvalidArgumentException('developer_id is required');
        }

        $state = uniqid(SalesChannelType::TYPE_WALMART.'_');
        $params = [
            'state' => $state,
            'nonce' => $state,
            'scope' => config('credentials.walmart.oauth_scopes'),
            'clientId' => $options['developer_id'],
            'redirectUri' => config('credentials.walmart.redirect_uri'),
        ];

        return ['state' => $state, 'url' => WalmartAPI::getOAuthURL($params)];
    }

    /**
     * Get user access token from code.
     *
     *
     * @return mixed
     */
    public function getAccessToken(string $code = '')
    {
        return $this->walmartSDK->getAccessToken();
    }

    /**
     * Refresh user access token.
     *
     *
     * @return mixed
     */
    public function refreshAccessToken(string $refresh_token)
    {
        // TODO: Implement refreshAccessToken() method.
    }

    /**
     * Get Orders from sales channel.
     *
     * @param  null  $options
     */
    public function getSalesOrders($options = null): Generator
    {
        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [
                'createdStartDate' => Carbon::now()->subDays(30)->format('Y-m-d'),
                //        'status'           => Enum::ORDER_STATUS_ACKNOWLEDGED,
            ];
        }

        /**
         * Set request pagination with initial page if not set.
         */
        if (! isset($options['limit'])) {
            $perPage = 10;
            $options['limit'] = $perPage;
        }

        do {
            $orders = $this->walmartSDK->getOrders($options);

            yield $orders;

            // Stop if nextCursor null.
            $hasNext = false;
            if (! empty($orders['list']['meta']['nextCursor'])) {
                $options = ['nextCursor' => $orders['list']['meta']['nextCursor']];

                $hasNext = true;
            }
        } while ($hasNext);
    }

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     */
    public function getProducts($options = null): Generator
    {
        // TODO: Implement getProducts() method.
    }
}
