<?php

namespace App\Integrations;

use App\Models\IntegrationInstance;
use App\SDKs\Starshipit\Model\Order;
use App\SDKs\Starshipit\Services\Orders;
use App\SDKs\Starshipit\Services\Tracking;
use App\SDKs\Starshipit\Starshipit as StarshipitSDK;
use App\SDKs\Starshipit\StarshipitException;
use App\SDKs\Starshipit\StarshipitResponse;
use Generator;
use InvalidArgumentException;

class Starshipit
{
    private $integrationInstance;

    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;

        $credentials = $integrationInstance->credentials;

        if (empty($credentials['apiKey']) || empty($credentials['subscriptionKey'])) {
            throw new InvalidArgumentException('apiKey and subscriptionKey are required');
        }

        StarshipitSDK::config($credentials['apiKey'], $credentials['subscriptionKey'], $integrationInstance->id);
    }

    /**
     * Create a new order.
     *
     *
     * @throws StarshipitException
     */
    public function submitOrder(Order $order): StarshipitResponse
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->create($order);
    }

    /**
     * Create new orders.
     *
     * @param  Order[]  $orders
     *
     * @throws StarshipitException
     */
    public function submitOrders(array $orders): StarshipitResponse
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->import($orders);
    }

    /**
     * Update order.
     *
     *
     * @throws StarshipitException
     */
    public function updateOrder(Order $order): StarshipitResponse
    {
        if (empty($order->order_id)) {
            throw new InvalidArgumentException('The order_id is required for update order');
        }

        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->update($order);
    }

    /**
     * Get order details.
     *
     * @param  int  $orderId
     *
     * @throws StarshipitException
     */
    public function getOrder(int|string $id, string $idType = 'order_id'): StarshipitResponse
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->get($id, $idType);
    }

    /**
     * Delete order details.
     *
     *
     * @throws StarshipitException
     */
    public function deleteOrder(int $orderId): StarshipitResponse
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->delete($orderId);
    }

    public function getTracking(string $orderNumber)
    {
        /** @var Tracking $trackingService */
        $trackingService = new Tracking();

        return $trackingService->get($orderNumber);
    }

    /**
     * Search printed order
     *
     *
     * @throws StarshipitException
     */
    public function searchOrders(array $options = []): Generator
    {
        $options['page'] = $options['page'] ?? 1;
        do {
            /** @var Orders $ordersService */
            $ordersService = new Orders();

            yield $response = $ordersService->search($options);

            $options['page']++;
        } while ($options['page'] <= $response->body['total_pages']);
    }

    public function getUnshippedOrders(array $options = [])
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->getUnshipped($options);
    }

    /**
     * @return Generator|StarshipitResponse[]
     *
     * @throws StarshipitException
     */
    public function getShippedOrders(array $options = []): Generator
    {
        $options['page'] = $options['page'] ?? 1;
        do {
            /** @var Orders $ordersService */
            $ordersService = new Orders();

            yield $response = $ordersService->getShipped($options);

            $options['page']++;
        } while ($options['page'] <= $response->body['total_pages']);
    }

    /**
     * Restore an archived order
     *
     *
     * @throws StarshipitException
     */
    public function restoreOrder(int $orderId): StarshipitResponse
    {
        /** @var Orders $ordersService */
        $ordersService = new Orders();

        return $ordersService->restore($orderId);
    }
}
