<?php

namespace App\Integrations;

use App\Enums\DownloadedBy;
use App\Helpers;
use App\Models\IntegrationInstance;
use App\Models\SalesChannelType;
use App\Models\Setting;
use App\Models\Shopify\ShopifyOrder;
use Carbon\Carbon;
use Exception;
use Generator;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use PHPShopify\AuthHelper;
use PHPShopify\Exception\ApiException;
use PHPShopify\Exception\CurlException;
use PHPShopify\ShopifySDK;

class Shopify extends Channel implements HasOAuth
{
    /**
     * Shopify Order Financial Status.
     */
    const FINANCIAL_STATUS_AUTHORIZED = 'authorized';

    const FINANCIAL_STATUS_PENDING = 'pending';

    const FINANCIAL_STATUS_PAID = 'paid';

    const FINANCIAL_STATUS_PARTIALLY_PAID = 'partially_paid';

    const FINANCIAL_STATUS_REFUNDED = 'refunded';

    const FINANCIAL_STATUS_VOIDED = 'voided';

    const FINANCIAL_STATUS_PARTIALLY_REFUNDED = 'partially_refunded';

    const FINANCIAL_STATUS_ANY = 'any';

    const FINANCIAL_STATUS_UNPAID = 'unpaid';

    /**
     * Shopify Order Fulfillment Status.
     */
    const FULFILMENT_QUERY_STATUS_SHIPPED = 'shipped';

    const FULFILMENT_QUERY_STATUS_PARTIAL = 'partial';

    const FULFILMENT_QUERY_STATUS_UNSHIPPED = 'unshipped';

    const FULFILMENT_QUERY_STATUS_ANY = 'any';

    const FULFILMENT_STATUS_FULFILLED = 'fulfilled';

    const FULFILMENT_STATUS_PARTIAL = 'partial';

    const FULFILMENT_STATUS_UNSHIPPED = 'null';

    const FULFILMENT_STATUS_CANCELED = 'restocked';

    /**
     * Shopify Order Status
     * This status for filter only, not returned with order object.
     */
    const ORDER_STATUS_OPEN = 'open';

    const ORDER_STATUS_CLOSED = 'closed';

    const ORDER_STATUS_CANCELLED = 'cancelled';

    const ORDER_STATUS_ANY = 'any';

    /**
     * Shopify SDK instance.
     *
     * @var ShopifySDK
     */
    private $shopifySDK;

    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    public static string $productChangesFilterField = 'updated_at_min';

    public function __construct(IntegrationInstance $integrationInstance)
    {
        /**
         * Check developer credentials.
         */
        $credentials = $integrationInstance->credentials;
        $credentials['api_key'] = $credentials['apiKey'];
        $credentials['api_secret_key'] = $credentials['password'];
        $credentials['shop'] = $credentials['shop_url'];

        if (array_diff(['api_key', 'api_secret_key'], array_keys($credentials))) {
            throw new \InvalidArgumentException('api_key and api_secret_key are required.');
        }

        $config = [
            'ApiVersion' => '2023-04',
            'ApiKey' => $credentials['api_key'],
            'Password' => $credentials['api_secret_key'],
        ];

        /**
         * Set client credentials if passed.
         */
        if (isset($credentials['access_token'])) {
            $config['Password'] = $credentials['access_token'];
        }

        if (isset($credentials['shop'])) {
            $config['ShopUrl'] = $credentials['shop'];
        }

        /**
         * Set config and create new instance of sdk.
         */
        //    ShopifySDK::config($config);
        $this->shopifySDK = new ShopifySDK($config);

        $this->integrationInstance = $integrationInstance;
    }

    public static function filterChangesByCreatedAt()
    {
        static::$productChangesFilterField = 'created_at_min';
    }

    public static function filterChangesByUpdatedAt()
    {
        static::$productChangesFilterField = 'updated_at_min';
    }

    /**
     * Generate OAuth user url.
     *
     *
     * @throws Exception
     */
    public function generateOAuthUrl(array $options = []): array
    {
        if (! array_key_exists('shop_name', $options)) {
            throw new Exception('shop_name is required for authentication request.');
        }

        ShopifySDK::$config['ShopUrl'] = $options['shop_name'].'.myshopify.com';
        ShopifySDK::setAdminUrl();

        $state = uniqid(SalesChannelType::TYPE_SHOPIFY.'_');
        // return oauth url.
        $url = AuthHelper::createAuthRequest(
            config('sales_channels_oauth.shopify.oauth_scopes'),
            urlencode(url(config('sales_channels_oauth.shopify.redirect_uri'), [], true)),
            $state,
            null,
            true
        );

        return ['state' => $state, 'url' => $url];
    }

    /**
     * Get user access token from code.
     *
     *
     * @throws Exception
     */
    public function getAccessToken(string $code = ''): string
    {
        if (! isset($_GET['shop']) && ! isset($_GET['code'])) {
            throw new Exception('shop and code are required for authentication request.');
        }
        ShopifySDK::$config['ShopUrl'] = $_GET['shop'];
        ShopifySDK::setAdminUrl();

        return AuthHelper::getAccessToken();
    }

    /**
     * Check sales channel credentials.
     *
     * @return mixed
     */
    public function checkCredentials()
    {
        // TODO: Implement checkCredentials() method.
    }

    /**
     * Refresh user access token.
     *
     *
     * @return mixed
     */
    public function refreshAccessToken(string $refresh_token)
    {
        // TODO: Implement refreshAccessToken() method.
    }

    /**
     * @param  null  $options
     *
     * @throws ApiException
     * @throws CurlException
     */
    public function getSalesOrders($options = null): Generator
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }

        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        // page_info: continue to the next pagination request
        if (isset($options['page_info'])) {
            $options = Arr::only($options, ['page_info', 'limit']);
        } elseif (! isset($options['ids'])) {
            if (! empty(Arr::only($options, ['created_after', 'created_before', 'updated_after', 'updated_before']))) {
                $options = $this->makeOrderSettings($options);
            } else {
                $options = array_merge(
                    $this->getIntegrationOrderSettings(),
                    $this->makeOrderSettings($options),
                );
            }
        } else {
            $options['status'] = 'any';
        }

        $resource = $this->shopifySDK->Order();

        do {
            yield $resource->get($options);

            yield $options = $resource->getNextPageParams();
        } while ($options);
    }

    public function getSalesOrdersNew($options = null)
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }

        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        // page_info: continue to the next pagination request
        if (isset($options['page_info'])) {
            $options = Arr::only($options, ['page_info', 'limit']);
        } elseif (! isset($options['ids'])) {
            if (! empty(Arr::only($options, ['created_after', 'created_before', 'updated_after', 'updated_before']))) {
                $options = $this->makeOrderSettings($options);
            } else {
                $options = array_merge(
                    $this->getIntegrationOrderSettings(),
                    $this->makeOrderSettings($options),
                );
            }
        } else {
            $options['status'] = 'any';
        }

        $resource = $this->shopifySDK->Order();

        $response = [];

        $response['orders'] = $resource->get($options);
        $response['nextPageParams'] = $resource->getNextPageParams();

        return $response;
    }

    public function getSalesOrderById($shopifyOrderId)
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }
        $resource = $this->shopifySDK->Order($shopifyOrderId);

        return $resource->get();
    }

    public function getOrderCounts($options = null)
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }

        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        // Merge integration order settings
        if (empty($options['ids'])) {
            /**
             * Set request pagination with initial page if not set.
             */
            if (! isset($options['limit'])) {
                $options['limit'] = 10;
            }

            $options = $this->getIntegrationOrderSettings();

            $options['status'] = 'any';
        }

        return $this->shopifySDK->Order->count();
    }

    public function getOrderTransactions($shopify_order_id)
    {
        $query = ' 
            query
            {
            orders(first: 1, query:"id:'.$shopify_order_id.'") {
                  edges {
                    node {
                      id
                      transactions {
                        id
                        kind
                        order {
                            id
                        }
                        status
                        fees {
                            amount {
                                amount
                                currencyCode
                            }
                            rateName
                            type
                        }
                        formattedGateway
                        gateway
                        receipt
                        processedAt
                        amount
                        amountSet {
                            presentmentMoney {
                                amount
                                currencyCode
                            }
                            shopMoney {
                                amount
                                currencyCode
                            }
                        }
                      }
                    }
                  }
                }
            }
            ';

        // password as access token
        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query);
    }

    public function getIntegrationOrderSettings(): array
    {
        $integrationSettings = $this->integrationInstance->integration_settings ?? [];
        if (empty($integrationSettings) || ! isset($integrationSettings['orders'])) {
            return [];
        }

        $settings = $integrationSettings['orders'];

        /** @see SKU-3459
         * 1st time use created_after
         * Then use the max of updated_at (returned from shopify) and use updated_after for the filter
         */
        $condition = 'created_after';
        $conditionDate = min($this->integrationInstance->open_start_date, $this->integrationInstance->closed_start_date);

        $lastOrder = ShopifyOrder::latestOrder($this->integrationInstance->id, ShopifyOrder::DOWNLOADED_BY_GET_ORDERS_JOB)->first();

        if ($lastOrder) {
            $condition = 'updated_after';
            $conditionDate = Carbon::parse($lastOrder->updated_at)->addSecond();
        }

        $settings[$condition] = $conditionDate;

        return $this->makeOrderSettings($settings);
    }

    private function makeOrderSettings(array $orderSettings): array
    {
        $options = [];

        if (isset($orderSettings['updated_after'])) {
            $options['updated_at_min'] = Carbon::parse($orderSettings['updated_after'], Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE))->endOfDay()->toIso8601String();
        }

        if (isset($orderSettings['updated_before'])) {
            $options['updated_at_max'] = Carbon::parse($orderSettings['updated_before'], Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE))->endOfDay()->toIso8601String();
        }

        if (isset($orderSettings['created_after'])) {
            $options['created_at_min'] = Carbon::parse($orderSettings['created_after'], Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE))->endOfDay()->toIso8601String();
        }

        if (isset($orderSettings['created_before'])) {
            $options['created_at_max'] = Carbon::parse($orderSettings['created_before'], Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE))->endOfDay()->toIso8601String();
        }

        if (isset($orderSettings['limit'])) {
            $options['limit'] = $orderSettings['limit'];
        }

        $options['status'] = 'any';
        $options['order'] = 'created_at asc';

        return $options;
    }

    /**
     * Get Products from sales channel.
     *
     * @param  null  $options
     */
    public function getProducts($options = null): Generator
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }

        /**
         * For initials options.
         */
        if (empty($options) || ! is_array($options)) {
            $options = [];
        }

        /**
         * Set request pagination with limit, if not specified.
         */
        if (! isset($options['limit'])) {
            $options['limit'] = 250;
        }

        /**
         * We only download active products.
         *
         * @see SKU-3493
         */
        $options['status'] = 'active';

        $dateField = self::$productChangesFilterField == 'updated_at_min' ? 'updated_at' : 'created_at';
        $lastProductUpdated = \App\Models\Shopify\ShopifyProduct::latestProduct($this->integrationInstance->id, DownloadedBy::Job, $dateField)->first();
        /** @see SKU-3116 get products after last product fetched (max updated_at) */
        if (empty($options['created_at_min']) && empty($options['created_at_max']) &&
          empty($options['updated_at_min']) && empty($options['updated_at_max']) &&
          $lastProductUpdated) {
            $options[self::$productChangesFilterField] = Carbon::parse($lastProductUpdated->{$dateField})->addSecond()->toISOString();
        }

        $resource = $this->shopifySDK->Product();

        do {
            yield $resource->get($options);
            $options = $resource->getNextPageParams();
        } while ($options);
    }

    public function getShippingServices()
    {
        if (empty(ShopifySDK::$config['Password']) || empty(ShopifySDK::$config['ShopUrl'])) {
            throw new \InvalidArgumentException('access_token and shop are required.');
        }

        return $this->shopifySDK->CarrierService()->get();
    }

    /**
     * Get Shopify Locations.
     *
     * @return array|false[]
     *
     * @throws ApiException
     * @throws CurlException
     */
    public function getLocations(): array
    {
        return $this->shopifySDK->Location->get();
    }

    /**
     * Create a webhook.
     *
     * @throws ApiException
     * @throws CurlException
     */
    public function createWebhook(array $options, string $api = 'rest'): array
    {
        if ($api == 'rest') {
            return $this->shopifySDK->Webhook->post($options);
        } elseif ($api == 'graphql') {
            $query = <<<'Query'
            mutation webhookSubscriptionCreate($topic: WebhookSubscriptionTopic!, $webhookSubscription: WebhookSubscriptionInput!) {
              webhookSubscriptionCreate(topic: $topic, webhookSubscription: $webhookSubscription) {
                userErrors {
                  field
                  message
                }
                webhookSubscription {
                  id
                }
              }
            }
            Query;

            $variables = [
                'topic' => $options['topic'],
                'webhookSubscription' => [
                    'callbackUrl' => $options['address'],
                ],
            ];

            // password as access token
            ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

            return $this->shopifySDK->GraphQL->post($query, null, null, $variables);
        }
    }

    public function getWebhooks(array $options = [])
    {
        return $this->shopifySDK->Webhook->get($options);
    }

    public function deleteWebhook($id)
    {
        return $this->shopifySDK->Webhook($id)->delete();
    }

    public function createFulfillment(array $options)
    {
        return $this->shopifySDK->Fulfillment()->post($options);
    }

    public function getFulfillmentOrder($orderId)
    {
        return $this->shopifySDK->Order($orderId)->FulfillmentOrder->get();
    }

    public function getFulfillments($orderId)
    {
        return $this->shopifySDK->Order($orderId)->Fulfillment()->get();
    }

    public function getBulkProductVariants(): array
    {
        $query = <<<'GRAPHQL'
            mutation {
                bulkOperationRunQuery(
                    query:""" 
                    {
                        productVariants(query: "product_status:active") {
                            edges {
                                node {
                                    id
                                }
                            }
                        }
                    }
                     """
                )
                {
                    bulkOperation {
                        id
                        status
                    }
                    userErrors {
                        field
                        message
                    }
                }
            }
            GRAPHQL;

        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query);
    }

    /**
     * @throws ApiException
     * @throws CurlException
     */
    public function getProductVariants($count, $cursor): array
    {
        $query = <<<GRAPHQL
            query FetchProductSamples(\$cursor: String) {
                productVariants(first: $count, after: \$cursor, query: "product_status:active") {
                    edges {
                        node {
                            id
                        }
                    }
                    pageInfo {
                        hasNextPage
                        endCursor
                    }
                }
            }
            GRAPHQL;

        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query, null, null, ['cursor' => $cursor]);
    }

    public function bulkAdjustQuantity(array $itemAdjustments, string $locationId)
    {
        $query = <<<'Query'
            mutation inventoryBulkAdjustQuantityAtLocation($inventoryItemAdjustments: [InventoryAdjustItemInput!]!, $locationId: ID!) {
              inventoryBulkAdjustQuantityAtLocation(inventoryItemAdjustments: $inventoryItemAdjustments, locationId: $locationId) {
                userErrors {
                  field
                  message
                }
                inventoryLevels {
                  available,
                  updatedAt,
                  item {id, sku, variant {id}},
                }
              }
            }
            Query;

        $variables = [
            'inventoryItemAdjustments' => $itemAdjustments,
            'locationId' => $locationId,
        ];
        // password as access token
        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query, null, null, $variables);
    }

    public function getInventoryItems(array $options = []): Generator
    {
        $resource = $this->shopifySDK->InventoryItem();
        do {
            yield $resource->get($options);
            // we don't need to yield the options here
            // because we don't use it to continue a failed job
            $options = $resource->getNextPageParams();
        } while ($options);
    }

    /**
     * @throws ApiException
     * @throws CurlException
     */
    public function getVariant(int $variantId): array
    {
        return $this->shopifySDK->ProductVariant($variantId)->get();
    }

    /**
     * Get transactions of orders by GraphQL
     *
     * @throws ApiException
     * @throws CurlException
     */
    public function getOrdersTransactions(array $orderIds, bool $bulk = false, ?string $filterQuery = null): array
    {
        $count = count($orderIds);
        $filterQuery = $filterQuery ?? 'id:'.implode(' OR ', $orderIds);

        $header = $bulk ? '
            mutation {
            bulkOperationRunQuery(
                query:""" 
        ' : '
        query
        ';

        $footer = $bulk ? '
            """
        )
        {
            bulkOperation {
                id
                status
            }
            userErrors {
                field
                message
            }
        }}' : '';

        $query = "$header 
            {
            orders(".(! $bulk ? 'first: '.$count.', ' : '')."query:\"$filterQuery\") {
                  edges {
                    node {
                      id
                      transactions {
                        id
                        kind
                        status
                        processedAt
                        fees {
                            amount {
                                amount
                                currencyCode
                            }
                            rateName
                            type
                        }
                        formattedGateway
                        gateway
                        receipt
                        amount
                        amountSet {
                            presentmentMoney {
                                amount
                                currencyCode
                            }
                            shopMoney {
                                amount
                                currencyCode
                            }
                        }
                      }
                    }
                  }
                }
            }
            $footer";

        // password as access token
        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query);
    }

    /**
     * Get transactions of orders by GraphQL
     *
     * @throws ApiException
     * @throws CurlException
     */
    public function getBulkOperationDataURL(string $admin_graphql_api_id): array
    {
        $query = <<<Query
            query {
              	node(id: "$admin_graphql_api_id") {
                  ... on BulkOperation {
                    url
                    objectCount
                    fileSize
                  }
                }
            }
            Query;

        // password as access token
        ShopifySDK::$config['AccessToken'] = ShopifySDK::$config['Password'];

        return $this->shopifySDK->GraphQL->post($query);
    }

    public static function getIdFromGID(string $gid)
    {
        $matches = [];
        preg_match('/^.*gid.*\\/([0-9]*)$/', $gid, $matches);

        if (empty($matches)) {
            //Log::debug('get ID from Invalid GID: '.$gid);
        }

        return $matches[1];
    }

    /**
     * return the seconds that should wait before retrying send the next Graphql request,
     * based on the query cost of the last Graphql response
     *
     * @return int seconds
     */
    public static function getGraphqlTimeToRetry(array $queryCost): int
    {
        $throttleStatus = $queryCost['throttleStatus'];
        if (($deficitCost = $throttleStatus['currentlyAvailable'] - $queryCost['actualQueryCost']) < 0) {
            return (int) ceil(abs($deficitCost) / $throttleStatus['restoreRate']);
        }

        return 0;
    }

    public function getDeletedEvents($options = [])
    {
        $resource = $this->shopifySDK->Event();

        do {
            yield $resource->get($options);

            yield $options = $resource->getNextPageParams();
        } while ($options);
    }

    public function setInventoryLevel(string $inventoryItemId, string $locationId, int $qtyAvailable): void
    {
        $resource = $this->shopifySDK->InventoryLevel();
        $resource->set([
            'inventory_item_id' => $inventoryItemId,
            'location_id' => $locationId,
            'available' => $qtyAvailable,
        ]);
    }

    public function getInventoryLevel(string $inventoryItemId, string $locationId): ?int
    {
        $resource = $this->shopifySDK->InventoryLevel();

        // Fetch the inventory level data
        $inventoryLevelData = $resource->get([
            'inventory_item_ids' => $inventoryItemId,
            'location_ids' => $locationId,
        ]);

        // Extract and return the available inventory quantity
        if (! empty($inventoryLevelData)) {
            return $inventoryLevelData[0]['available'];
        }

        return null; // Return null if no inventory data is available
    }
}
