<?php

namespace App\Integrations;

use App\Models\IntegrationInstance;
use App\SDKs\ShipStation\Model\Order;
use App\SDKs\ShipStation\Model\ShipStationWebhook;
use App\SDKs\ShipStation\Requests\ListOrdersRequest;
use App\SDKs\ShipStation\Requests\ListShipmentsRequest;
use App\SDKs\ShipStation\Requests\WebhooksShipmentsRequest;
use App\SDKs\ShipStation\Responses\BulkCreateUpdateOrdersResponse;
use App\SDKs\ShipStation\Responses\CreateUpdateOrdersResponse;
use App\SDKs\ShipStation\Responses\ListOrdersResponse;
use App\SDKs\ShipStation\Services\Carriers;
use App\SDKs\ShipStation\Services\Orders;
use App\SDKs\ShipStation\Services\Shipments;
use App\SDKs\ShipStation\Services\Warehouses;
use App\SDKs\ShipStation\Services\ShipStationWebhookResult;
use App\SDKs\ShipStation\Services\Webhooks;
use App\SDKs\ShipStation\ShipStation as ShipStationSDK;
use App\SDKs\ShipStation\ShipStationException;
use Generator;
use InvalidArgumentException;

class ShipStation
{
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $credentials = $integrationInstance->credentials;

        if (empty($credentials['apiKey']) || empty($credentials['apiSecret'])) {
            throw new InvalidArgumentException('apiKey and apiSecret are required');
        }

        ShipStationSDK::config($credentials['apiKey'], $credentials['apiSecret']);
    }

    /**
     * Get carriers from ShipStation.
     */
    public function getCarriers(): array
    {
        $carriersService = new Carriers();

        return $carriersService->getList();
    }

    /**
     * Get shipping services of the carrier from ShipStation.
     */
    public function getShippingServices(string $carrierCode): array
    {
        $carriersService = new Carriers();

        return $carriersService->getServices($carrierCode);
    }

    public function getOrder(string $orderId): array
    {
        $ordersService = new Orders();

        return $ordersService->view($orderId);
    }

    public function getSalesOrders(?ListOrdersRequest $listOrdersRequest = null): Generator|ListOrdersResponse
    {
        if (! $listOrdersRequest) {
            $listOrdersRequest = ListOrdersRequest::defaultInstance();
        }

        if (empty($listOrdersRequest->page)) {
            $listOrdersRequest->page = 1;
        }

        $ordersService = new Orders();

        do {
            yield $response = $ordersService->getList($listOrdersRequest);

            $listOrdersRequest->page += 1;
        } while ($response->page != $response->pages);
    }

    /**
     * @param  Order|Order[]  $orders
     *
     * @throws ShipStationException
     */
    public function submitOrders($orders): BulkCreateUpdateOrdersResponse
    {
        if ($orders instanceof Order) {
            $orders = [$orders];
        }

        $ordersService = new Orders();

        return $ordersService->bulkCreateUpdate($orders);
    }

    public function submitOrder(Order $order): CreateUpdateOrdersResponse
    {
        $ordersService = new Orders();

        return $ordersService->create($order);
    }

    public function deleteOrders(string $orderId): array
    {
        $ordersService = new Orders();

        customlog('SKU-6191', 'Deleting shipstation fulfillment '.$orderId, [
            'debug' => debug_pretty_string(),
        ], 7);

        return $ordersService->delete($orderId);
    }

    /**
     * @throws ShipStationException
     */
    public function getShipments(?ListShipmentsRequest $listShipmentsRequest = null): Generator|array
    {
        if (! $listShipmentsRequest) {
            $listShipmentsRequest = ListShipmentsRequest::defaultInstance();
        }

        if (empty($listShipmentsRequest->page)) {
            $listShipmentsRequest->page = 1;
        }

        $shipmentsService = new Shipments();

        do {
            yield $response = $shipmentsService->getList($listShipmentsRequest);

            $listShipmentsRequest->page += 1;

            yield $listShipmentsRequest;
        } while ($response->page != $response->pages && $response->pages != 0);
    }

    /**
     * @throws ShipStationException
     */
    public function getWebhooksResult(ShipStationWebhook $webhook, ?WebhooksShipmentsRequest $webhooksShipmentsRequest = null): Generator|array
    {
        if (! $webhooksShipmentsRequest) {
            $webhooksShipmentsRequest = WebhooksShipmentsRequest::defaultInstance();
        }

        if (empty($webhooksShipmentsRequest->page)) {
            $webhooksShipmentsRequest->page = 1;
        }
        $webhooksResult = new ShipStationWebhookResult();

        do {
            $response = $webhooksResult->getList($webhook, $webhooksShipmentsRequest);

            yield $response;

            $webhooksShipmentsRequest->page += 1;
        } while ($response->page != $response->pages);
    }

    /**
     * @throws ShipStationException
     */
    public function createWebhook(ShipStationWebhook $webhook): ShipStationWebhook
    {
        $webhooksService = new Webhooks();

        return $webhooksService->create($webhook);
    }

    /**
     * @throws ShipStationException
     */
    public function getWebhooks(): array
    {
        $webhooksService = new Webhooks();

        return $webhooksService->list();
    }

    /**
     * @throws ShipStationException
     */
    public function deleteWebhook(int $shipstation_webhook_id): bool|array
    {
        $webhooksService = new Webhooks();

        return $webhooksService->delete($shipstation_webhook_id);
    }

    public function getWarehouses()
    {
        return (new Warehouses())->getList();
    }
}
