<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 12/14/20
 * Time: 2:42 PM.
 */

namespace App\Importers;

use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Repositories\SupplierInventoryRepository;
use App\Response;
use Illuminate\Support\Arr;

/**
 * Class SupplierInventoryExporter.
 */
class SupplierInventoryImporter extends FileImporter
{
    private $supplierId;

    /**
     * @var array
     */
    protected $expectedColumns = [
        'inventory_id',
        'supplier_sku',
        'quantity',
        'in_stock',
        'eta',
    ];

    /**
     * @var array
     */
    protected $requiredColumns = ['inventory_id'];

    /**
     * @var SupplierInventoryRepository
     */
    private $inventories;

    /**
     * @var string
     */
    protected $idColumn = 'inventory_id';

    /**
     * SupplierInventoryExporter constructor.
     */
    public function __construct($taskId, string $filePath, SupplierInventoryRepository $inventories, $supplierId)
    {
        parent::__construct($taskId, $filePath);
        $this->supplierId = $supplierId;
        $this->inventories = $inventories;
    }

    protected function importRow(array $row): void
    {
        $inventory = $this->getInventory($row);
        if (! $inventory) {
            return;
        }
        // Update the inventory
        $row['source'] = SupplierInventory::SOURCE_USER;
        $inventory->update($row);

        // Set supplier sku if present
        if (isset($row['supplier_sku']) && $inventory->product_id) {
            $supplierProduct = $inventory->supplier->supplierProducts()->where('product_id', $inventory->product_id)->first();
            if ($supplierProduct) {
                $supplierProduct->update([
                    'supplier_sku',
                    $row['supplier_sku'],
                ]);
            }
        }
    }

    private function getInventory(array $params): ?SupplierInventory
    {
        if (! empty($params['inventory_id'])) {
            // Attempt to find inventory by id
            $inventory = $this->inventories->findBy([
                'id' => $params['inventory_id'],
                'supplier_id' => $this->supplierId,
            ]);
            if ($inventory) {
                return $inventory;
            } else {
                $this->validationErrors[$params['supplier_sku'] ?? $params['inventory_id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $params['inventory_id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($params, ['inventory_id', 'supplier_sku']));
                $this->taskStatus->addErrorMessage("Skipping id: The id {$params['inventory_id']} doesn't exist in SKU");

                return null;
            }
        }

        // Couldn't match supplier and inventory id,
        // we attempt to match by supplier_sku and supplier id
        if (! empty($params['supplier_sku'])) {
            $inventory = $this->inventories->findBy([
                'supplier_sku' => $params['supplier_sku'],
                'supplier_id' => $this->supplierId,
            ]);
            if ($inventory) {
                return $inventory;
            }
        }

        // We create new inventory
        $params['supplier_id'] = $this->supplierId;
        $params['warehouse_id'] = $params['warehouse_id'] ?? Supplier::with([])->findOrFail($this->supplierId)->defaultWarehouse->id;

        return $this->inventories->createInventory($params);
    }
}
