<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exceptions\ImportFailedException;
use App\Importers\DataImporter;
use App\Jobs\ImportSupplierInventory;
use App\Mail\ImportCompletedMail;
use App\Models\SupplierInventory;
use Illuminate\Bus\PendingBatch;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Mail;
use Throwable;

class SupplierInventoryDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'quantity',
        'in_stock',
    ];

    /**
     * SupplierDataImporter constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);
        $this->expectedColumns = collect(SupplierInventory::getExportableFields())->map(function ($field) {
            return $field['exported_as'];
        })->values()->toArray();
    }

    public function importableFields(): array
    {
        $fields = collect(SupplierInventory::getExportableFields())->filter(function ($field) {
            return $field['importable'] == true;
        })->toArray();

        return array_values(collect($fields)->map(function ($field) {
            return [
                'key' => $field['exported_as'],
                'value' => $field['label'],
            ];
        })->toArray());
    }

    /**
     * @throws Throwable
     * @throws ImportFailedException
     */
    protected function importByRow(): bool
    {
        $rows = [];
        $batchSize = 1000;
        $batch = null;

        foreach ($this->yieldRecords() as $record) {
            $rows[] = $record;

            if (count($rows) == $batchSize) {
                $batch = $this->scheduleBatch($rows, $batch);
                $rows = [];
            }
        }

        if (! $batch && empty($rows)) {
            return true; // Nothing to import.
        }

        if (! $batch && count($rows) <= DataImporter::MAX_SYNCHRONOUS_IMPORT_SIZE) {
            // We import synchronously
            dispatch_sync(
                new ImportSupplierInventory($rows, $this->meta)
            );

            return true;
        }

        // At this point, we do have a lot of data so we
        // use the batching process.

        // Process any remaining batch.
        if (! empty($rows)) {
            $batch = $this->scheduleBatch($rows, $batch);
        }

        $batch->then(function ($_) {
            // Email the user about the import
            Mail::to($this->user)->queue(
                new ImportCompletedMail($this->user, SupplierInventory::class)
            );
        })
            ->name('Supplier Inventory Importer')
            ->dispatch();

        return true; // Success
    }

    private function scheduleBatch(array $rows, ?PendingBatch $batch = null): PendingBatch
    {
        // We import the batch in another job
        if ($batch) {
            return $batch->add(new ImportSupplierInventory($rows, $this->meta));
        }

        return Bus::batch([new ImportSupplierInventory($rows, $this->meta)]);
    }

    public function getSupplierId(): ?int
    {
        return $this->meta['supplier_id'] ?? null;
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'quantity' => Exporter::makeExportableField('quantity', true),
            'in_stock' => Exporter::makeExportableField('in_stock', true),
            'sku' => Exporter::makeExportableField('sku', true, 'SKU'),
            'supplier_sku' => Exporter::makeExportableField('supplier_sku', true),
            'supplier_id' => Exporter::makeExportableField('supplier_id', false),
            'eta' => Exporter::makeExportableField('eta', true, 'ETA'),
            'source' => Exporter::makeExportableField('source', false),
            'product_name' => Exporter::makeExportableField('product_name', false),
            'product_id' => Exporter::makeExportableField('product_id', true, 'Product ID'),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name', false),
            'warehouse.id' => Exporter::makeExportableField('warehouse_id', false, 'Warehouse ID'),
        ];
    }

    public function sendsImportEmail(): bool
    {
        return true;
    }

    protected function importRow(array $row)
    {
        return null;
    }
}
