<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Supplier;
use App\Repositories\WarehouseRepository;
use App\Response;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class SupplierDataImporter.
 */
class SupplierDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
    ];

    /**
     * @var WarehouseRepository
     */
    protected $warehouses;

    /**
     * SupplierDataImporter constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);
        $this->warehouses = app()->make(WarehouseRepository::class);
    }

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given supplier or update it if possible
            // We attempt to find by id or name
            if (! empty($row['id'])) {
                $supplier = Supplier::with([])->find($row['id']);
                if (! $supplier) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $supplier = Supplier::with([])->where('name', $row['name'])->firstOrNew();
            }

            $supplier->fill($row)->save();

            // Create address if data available.
            $addressFields = [];
            foreach ($row as $key => $value) {
                if (Str::startsWith($key, 'address.')) {
                    $parts = explode('.', $key);
                    $addressFields[$parts[1]] = $value;
                }
            }
            if (! empty($addressFields)) {
                $supplier->updateOrCreateAddress($addressFields);
            }

            // If the supplier was newly created, we create
            // a default warehouse for the supplier.
            if ($supplier->wasRecentlyCreated) {
                $this->warehouses->createDefaultWarehouseForSupplier($supplier);
            }
        });
    }

    /**
     * Fields for export.
     */
    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'name' => Exporter::makeExportableField('name', true, 'Supplier Name'),
            'company_name' => Exporter::makeExportableField('company_name'),
            'primary_contact_name' => Exporter::makeExportableField('primary_contact_name'),
            'email' => Exporter::makeExportableField('email'),
            'purchase_order_email' => Exporter::makeExportableField('purchase_order_email'),
            'website' => Exporter::makeExportableField('website', true, 'Supplier Website'),
            'leadtime' => Exporter::makeExportableField('leadtime'),
            'minimum_order_quantity' => Exporter::makeExportableField('minimum_order_quantity'),
            'minimum_purchase_order' => Exporter::makeExportableField('minimum_purchase_order'),
            'phone' => Exporter::makeExportableField('address.phone', 'Phone Number'),
            'fax' => Exporter::makeExportableField('address.fax', true, 'Address Fax'),
            'address1' => Exporter::makeExportableField('address.address1', true, 'Address Line 1'),
            'address2' => Exporter::makeExportableField('address.address2', true, 'Address Line 2'),
            'address3' => Exporter::makeExportableField('address.address3', true, 'Address Line 3'),
            'city' => Exporter::makeExportableField('address.city', true, 'City'),
            'province' => Exporter::makeExportableField('address.province', true, 'Province'),
            'province_code' => Exporter::makeExportableField('address.province_code', true, 'Province Code'),
            'zip' => Exporter::makeExportableField('address.zip', true, 'Zip'),
            'country' => Exporter::makeExportableField('address.country', true, 'Country'),
            'country_code' => Exporter::makeExportableField('address.country_code', true, 'Country Code'),
            'label' => Exporter::makeExportableField('address.label', true, 'Address Label'),
            'default_pricing_tier_id' => Exporter::makeExportableField('default_pricing_tier_id', true, 'Default Pricing Tier ID'),
            'default_warehouse_id' => Exporter::makeExportableField('default_warehouse_id', true, 'Default Warehouse ID'),
            'default_store_id' => Exporter::makeExportableField('default_store_id', true, 'Default Store ID'),
        ];
    }
}
