<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Address;
use App\Models\Store;
use App\Response;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class StoreDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
        'email',
    ];

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given store or update it if possible
            // We attempt to find by id or name and email
            if (! empty($row['id'])) {
                $store = Store::with([])->find($row['id']);
                if (! $store) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $store = Store::with([])->where('name', $row['name'])
                    ->where('email', $row['email'])
                    ->firstOrNew();
            }

            $store = $store->fill($row);
            $store->save();

            // Handle the address of the store
            if ($store->wasRecentlyCreated || ! $store->address) {
                $address = new Address();
            } else {
                $address = $store->address;
            }
            $address->fill($row);
            $address->save();

            $store->address()->associate($address);
        });
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'name' => Exporter::makeExportableField('name'),
            'email' => Exporter::makeExportableField('email'),
            'company_name' => Exporter::makeExportableField('company_name', true),
            'label' => Exporter::makeExportableField('label', true, 'Address Label'),
            'phone' => Exporter::makeExportableField('phone', true, 'Phone Number'),
            'fax' => Exporter::makeExportableField('fax', true),
            'address1' => Exporter::makeExportableField('address1', true, 'Address Line 1'),
            'address2' => Exporter::makeExportableField('address2', true, 'Address Line 2'),
            'address3' => Exporter::makeExportableField('address3', true, 'Address Line 3'),
            'city' => Exporter::makeExportableField('city', true),
            'province' => Exporter::makeExportableField('province', true),
            'province_code' => Exporter::makeExportableField('province_code', true),
            'zip' => Exporter::makeExportableField('zip', true),
            'country' => Exporter::makeExportableField('country', true),
            'country_code' => Exporter::makeExportableField('country_code', true),
        ];
    }
}
