<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Response;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class ShippingMethodDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
        'carrier_name',
        'type',
    ];

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given shipping method or update it if possible
            // We attempt to find by id or name and type
            if (! empty($row['id'])) {
                $method = ShippingMethod::with([])->find($row['id']);
                if (! $method) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $method = ShippingMethod::with([])->where('name', $row['name'])
                    ->where('type', $row['type'])
                    ->firstOrNew();
            }

            if (! $method->exists) {
                $carrier = ShippingCarrier::with([])->where('name', $row['carrier_name'])->first();
                if (! $carrier) {
                    $carrier = new ShippingCarrier;
                    $carrier->name = $row['carrier_name'];
                    $carrier->save();
                }

                $method->shipping_carrier_id = $carrier->id;
            }

            $method->name = $row['name'];
            $method->save();
        });
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'method_name' => Exporter::makeExportableField('name'),
            'carrier_name' => Exporter::makeExportableField('carrier_name', 'Shipping Carrier Name'),
            'delivery_min' => Exporter::makeExportableField('delivery_min', true, 'Min Delivery Days'),
            'delivery_max' => Exporter::makeExportableField('delivery_max', true, 'Max Delivery Days'),
            'type' => Exporter::makeExportableField('type', true, 'Type (International or Domestic or Both)'),
        ];
    }
}
