<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\ShippingMethod;
use App\Models\Warehouse;
use App\Repositories\WarehouseRepository;
use App\Response;
use App\Services\SalesOrder\ApproveSalesOrderService;
use App\Services\SalesOrder\SalesOrderValidator;
use App\Validator as AppValidator;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class SalesOrderDataImporter extends DataImporter
{
    /**
     * @var SalesOrderValidator
     */
    private $validator;

    /**
     * @var WarehouseRepository
     */
    private $warehouses;

    /**
     * SalesOrderDataImporter constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);

        $this->validator = new SalesOrderValidator;
        $this->warehouses = app()->make(WarehouseRepository::class);
    }

    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            if (! empty($row['id'])) {
                $salesOrder = SalesOrder::query()->find($row['id']);
                if (! $salesOrder) {
                    $this->validationErrors[$row['sales_order_number'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'sales_order_number']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $salesOrder = SalesOrder::query()
                    ->where('sales_order_number', $row['sales_order_number'] ?? null)
                    ->when(! empty($row['item_id']), function (Builder $builder) use ($row) {
                        $builder->orWhereHas('salesOrderLines', function (Builder $builder) use ($row) {
                            $builder->where('id', $row['item_id']);
                        });
                    })->first();
            }

            if (! $salesOrder) {
                if (! $this->isDataValid($row, SalesOrderValidator::SALES_ORDER_ACTION_CREATE)) {
                    $this->taskStatus->addErrorMessage("Skipping sales order number: {$row['sales_order_number']} due to validation error.");

                    return;
                } // We skip the row since data isn't valid

                $salesOrder = $this->createSalesOrder($row);
            }

            if (! $salesOrder->wasRecentlyCreated) {
                // Validate for update.
                if (! $this->isDataValid($row, SalesOrderValidator::SALES_ORDER_ACTION_UPDATE, $salesOrder)) {
                    $this->taskStatus->addErrorMessage("Skipping sales order number: {$row['sales_order_number']} due to validation error.");

                    return;
                } // We skip the row since data isn't valid
                $this->updateSalesOrder($salesOrder, $row);
            }
        });
    }

    private function isDataValid(array $row, string $action, ?SalesOrder $salesOrder = null): bool
    {
        $validation = Validator::make(
            $row,
            $this->validator
                ->withAction($action, $salesOrder)
                ->rules(),
            $this->validator->messages()
        );

        if (! $validation->fails()) {
            $validation = $validation->after(function (AppValidator $validator) {
                $this->validator->withValidator($validator);
            });
        }

        if ($validation->fails() && ! $this->inBackground) {
            $this->validationErrors[$row['sales_order_number']] = $validation->errors()->toArray();
        }

        return ! $validation->fails();
    }

    /**
     * Updates the given sales order with the inputs.
     *
     *
     * @throws \Throwable
     */
    private function updateSalesOrder(SalesOrder $salesOrder, array $inputs): SalesOrder
    {
        $salesOrder->update($inputs);

        if ($salesOrder->order_status !== SalesOrder::STATUS_CLOSED) {
            $salesOrder->setSalesOrderLines(array_key_exists('sales_order_lines', $inputs) ? $inputs['sales_order_lines'] : false);
        }

        if ((isset($inputs['order_status']) && $inputs['order_status'] == SalesOrder::STATUS_OPEN) ||
            ($salesOrder->order_status === SalesOrder::STATUS_OPEN && array_key_exists('sales_order_lines', $inputs))
        ) {
            $salesOrder->approve(ApproveSalesOrderService::APPROVE_DROPSHIP_UPDATE_ONLY);
        }

        return $salesOrder;
    }

    /**
     * Creates a new sales order.
     *
     *
     * @throws \Throwable
     */
    private function createSalesOrder(array $inputs): SalesOrder
    {
        $salesOrder = new SalesOrder($inputs);
        $salesOrder->setCustomer($inputs['customer'] ?? null);
        $salesOrder->setShippingAddress($inputs['shipping_address'] ?? null);
        $salesOrder->setBillingAddress($inputs['billing_address'] ?? null);

        $salesOrder->save();

        $salesOrder->setSalesOrderLines($inputs['sales_order_lines'] ?? null);

        if ($inputs['order_status'] != SalesOrder::STATUS_DRAFT) {
            $salesOrder->approve($inputs['order_status'] == SalesOrder::STATUS_CLOSED);
        }

        if ($inputs['order_status'] == SalesOrder::STATUS_CLOSED) {
            $salesOrder->fulfilled_at = $salesOrder->order_date;

            $salesOrder->fullyFulfill($inputs['tracking_number'] ?? null, true);
        }

        return $salesOrder;
    }

    protected function importByRow(): bool
    {
        // We load the records
        if (! isset($this->records)) {
            $this->loadRecords();
        }

        // We aggregate the sales order data based on sales order number
        // and prepare them for creation/update.
        $records = collect($this->records);
        $salesOrders = [];
        $uniqueOrderNumbers = $records->unique('sales_order_number')->pluck('sales_order_number');
        $uniqueOrderNumbers->each(function ($salesOrderNumber, $index) use (&$salesOrders, $records) {
            // Make the sales order data.
            $rows = array_values($records->where('sales_order_number', $salesOrderNumber)->toArray());
            $row = $rows[0];
            $order = [
                'id' => $row['id'] ?? null,
                'sales_order_number' => $salesOrderNumber,
                'store_name' => $row['store_name'] ?? null,
                'order_status' => $row['order_status'] ?? SalesOrder::STATUS_DRAFT,
                'payment_status' => $row['payment_status'] ?? null,
                'currency_code' => $row['currency_code'] ?? null,
                'order_date' => $row['order_date'] ?? now(),
                'ship_by_date' => $row['ship_by_date'] ?? null,
                'deliver_by_date' => $row['deliver_by_date'] ?? null,
                'payment_date' => $row['payment_date'] ?? null,
            ];

            if (! empty($row['shipping_method'])) {
                $order['requested_shipping_method'] = $row['shipping_method'];
                /* @deprecated
                $shippingMethod = ShippingMethod::with([])->where('full_name', e($row['shipping_method']))->first();
                if ($shippingMethod) {
                $order['shipping_method_id'] = $shippingMethod->id;
                }*/
            }

            $customerKeys = array_filter(array_keys($row), function ($key) {
                return Str::startsWith($key, 'customer_');
            });

            $customerInfo = [];
            foreach ($customerKeys as $customerKey) {
                $customerInfo[str_replace('customer_', '', $customerKey)] = $row[$customerKey];
            }

            if (! empty($customerInfo)) {
                $customerInfo['address1'] = $customerInfo['address'] ?? null;

                $order['customer'] = $order['shipping_address'] = $order['billing_address'] = $customerInfo;
            }

            // Sales order lines
            $lineKeys = array_filter(array_keys($row), function ($key) {
                return Str::startsWith($key, 'item_') || in_array($key, ['is_product', 'warehouse_id', 'warehouse_name']);
            });

            $linesInfo = [];
            foreach ($rows as $key => $row) {
                $lineInfo = [];
                foreach ($lineKeys as $lineKey) {
                    $product = null;

                    // Skip lines with non-existent sku's
                    if ($lineKey === 'item_sku') {
                        $product = Product::with([])->where('sku', $row['item_sku'])->first();
                        if (! $product) {
                            continue;
                        }
                        $lineInfo['product_id'] = $product->id;
                        $lineInfo['is_product'] = true;
                        $lineInfo['warehouse_id'] = $this->getLineWarehouseId($row, $product);
                    }

                    $lineInfo[str_replace('item_', '', $lineKey)] = $row[$lineKey];
                    $lineInfo['description'] = $lineInfo['name'] ?? null;
                }
                $linesInfo[] = $lineInfo;
            }

            if (! empty($linesInfo)) {
                $order['sales_order_lines'] = $linesInfo;
            }

            $salesOrders[] = $order;
        });

        // Update the records
        $this->records = $salesOrders;

        return parent::importByRow();
    }

    private function getLineWarehouseId(array $row, Product $product): ?int
    {
        if (! empty($row['warehouse_id'])) {
            return $row['warehouse_id'];
        }

        if (! empty($row['warehouse_name'])) {
            $warehouse = Warehouse::with([])->where('name', e($row['warehouse_name']))->first();
            if ($warehouse) {
                return $warehouse->id;
            }
        }

        // Fall back to the first warehouse in the warehouse priority list
        return $this->warehouses->getPriorityWarehouseIdForProduct($product, $row['item_quantity']);
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'sales_order_number' => Exporter::makeExportableField('sales_order_number'),
            'order_date' => Exporter::makeExportableField('order_date', true),
            'integration_image' => Exporter::makeExportableField('integration_image', false),
            'customer_name.name' => Exporter::makeExportableField('customer_name', true),
            'customer_email' => Exporter::makeExportableField('customer_email'),
            'customer_company' => Exporter::makeExportableField('customer_company'),
            'customer_phone' => Exporter::makeExportableField('customer_phone'),
            'customer_address' => Exporter::makeExportableField(
                'customer_address',
                true,
                'Customer Address Line 1'
            ),
            'customer_city' => Exporter::makeExportableField('customer_city'),
            'customer_province' => Exporter::makeExportableField('customer_province'),
            'customer_province_code' => Exporter::makeExportableField('customer_province_code'),
            'customer_zip' => Exporter::makeExportableField('customer_zip'),
            'customer_country' => Exporter::makeExportableField('customer_country'),
            'customer_country_code' => Exporter::makeExportableField('customer_country_code'),
            'currency_code' => Exporter::makeExportableField('currency_code'),
            'order_status' => Exporter::makeExportableField('order_status', true),
            'fulfillment_status' => Exporter::makeExportableField('fulfillment_status', false),
            'payment_status' => Exporter::makeExportableField('payment_status', false),
            'product_total' => Exporter::makeExportableField('product_total', false),
            'total' => Exporter::makeExportableField('total', false),
            'sales_channel_name' => Exporter::makeExportableField('sales_channel_name', false),
            'store.name' => Exporter::makeExportableField('store_name', true),
            'fulfilled_at' => Exporter::makeExportableField('fulfilled_at', false),
            'deliver_by_date' => Exporter::makeExportableField('deliver_by_date', true),
            'ship_by_date' => Exporter::makeExportableField('ship_by_date', true),
            'shipping_method' => Exporter::makeExportableField('shipping_method', true),
            'fulfillment_channel' => Exporter::makeExportableField('fulfillment_channel', false),
            'sales_order_line_id' => Exporter::makeExportableField('item_id', true, 'Item ID'),
            'sales_channel_line_id' => Exporter::makeExportableField(
                'sales_channel_line_id',
                false
            ),
            'unit_cost' => Exporter::makeExportableField('unit_cost', false),
            'item_quantity' => Exporter::makeExportableField('item_quantity', true),
            'unfulfilled_quantity' => Exporter::makeExportableField(
                'item_quantity_unfulfilled',
                true
            ),
            'item_sku.sku' => Exporter::makeExportableField('item_sku', true),
            'item_price' => Exporter::makeExportableField('item_amount', true),
            'item_name' => Exporter::makeExportableField('item_name', true),
            'item_nominal_code' => Exporter::makeExportableField(
                'item_nominal_code_name',
                true
            ),
            'item_discount' => Exporter::makeExportableField('item_discount', true),
            'item_tax' => Exporter::makeExportableField('item_tax', true),
            'is_product' => Exporter::makeExportableField('is_product', true),
            'is_bundle' => Exporter::makeExportableField('is_bundle', true),
            'is_variation' => Exporter::makeExportableField('is_variation', true),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name', true),
            'total_revenue' => Exporter::makeExportableField(
                'total_revenue',
                true
            ),
            'has_backorder' => Exporter::makeExportableField('has_backorder', false),
            'packing_slip_printed_at' => Exporter::makeExportableField(
                'packing_slip_printed_at',
                false
            ),
            'canceled_at' => Exporter::makeExportableField('canceled_at', false),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
            'sales_order_shipping_address.name' => Exporter::makeExportableField('sales_order_shipping_address_name', true),
            'sales_order_shipping_address.email' => Exporter::makeExportableField('sales_order_shipping_address_email', true),
            'sales_order_shipping_address.company' => Exporter::makeExportableField('sales_order_shipping_address_company', true),
            'sales_order_shipping_address.phone' => Exporter::makeExportableField('sales_order_shipping_address_phone', true),
            'sales_order_shipping_address.address1' => Exporter::makeExportableField('sales_order_shipping_address_address1', true),
            'sales_order_shipping_address.address2' => Exporter::makeExportableField('sales_order_shipping_address_address2', true),
            'sales_order_shipping_address.address3' => Exporter::makeExportableField('sales_order_shipping_address_address3', true),
            'sales_order_shipping_address.city' => Exporter::makeExportableField('sales_order_shipping_address_city', true),
            'sales_order_shipping_address.province' => Exporter::makeExportableField('sales_order_shipping_address_province', true),
            'sales_order_shipping_address.province_code' => Exporter::makeExportableField('sales_order_shipping_address_province_code', true),
            'sales_order_shipping_address.zip' => Exporter::makeExportableField('sales_order_shipping_address_zip', true),
            'sales_order_shipping_address.country' => Exporter::makeExportableField('sales_order_shipping_address_country', true),
            'sales_order_shipping_address.country_code' => Exporter::makeExportableField('sales_order_shipping_address_country_code', true),

            'sales_order_billing_address.name' => Exporter::makeExportableField('sales_order_billing_address_name', true),
            'sales_order_billing_address.email' => Exporter::makeExportableField('sales_order_billing_address_email', true),
            'sales_order_billing_address.company' => Exporter::makeExportableField('sales_order_billing_address_company', true),
            'sales_order_billing_address.phone' => Exporter::makeExportableField('sales_order_billing_address_phone', true),
            'sales_order_billing_address.address1' => Exporter::makeExportableField('sales_order_billing_address_address1', true),
            'sales_order_billing_address.address2' => Exporter::makeExportableField('sales_order_billing_address_address2', true),
            'sales_order_billing_address.address3' => Exporter::makeExportableField('sales_order_billing_address_address3', true),
            'sales_order_billing_address.city' => Exporter::makeExportableField('sales_order_billing_address_city', true),
            'sales_order_billing_address.province' => Exporter::makeExportableField('sales_order_billing_address_province', true),
            'sales_order_billing_address.province_code' => Exporter::makeExportableField('sales_order_billing_address_province_code', true),
            'sales_order_billing_address.zip' => Exporter::makeExportableField('sales_order_billing_address_zip', true),
            'sales_order_billing_address.country' => Exporter::makeExportableField('sales_order_billing_address_country', true),
            'sales_order_billing_address.country_code' => Exporter::makeExportableField('sales_order_billing_address_country_code', true),
        ];
    }
}
