<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exceptions\ActionUnavailableTemporaryException;
use App\Helpers;
use App\Http\Requests\ReceiptPurchaseOrderRequest;
use App\Http\Requests\StorePurchaseInvoice;
use App\Importers\DataImporter;
use App\Models\Currency;
use App\Models\Product;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\Setting;
use App\Models\ShippingMethod;
use App\Models\Warehouse;
use App\Response;
use App\Services\PurchaseOrder\PurchaseOrderValidator;
use App\Services\PurchaseOrder\ShipmentManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class PurchaseOrderDataImporter extends DataImporter
{
    /**
     * @var PurchaseOrderValidator
     */
    private $validator;

    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);

        $this->validator = new PurchaseOrderValidator;
    }

    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            if (! empty($row['id'])) {
                $purchaseOrder = PurchaseOrder::query()->whereRelation('purchaseOrderLines', 'id', $row['id'])->first();
                if (! $purchaseOrder) {
                    $this->validationErrors[$row['purchase_order_number'] ?? $row['id']] = Response::getError("ID {$row['id']} doesn't exist in SKU purchase order lines, review file and import again.", Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'purchase_order_number']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $purchaseOrder = PurchaseOrder::query()
                    ->where('purchase_order_number', $row['purchase_order_number'] ?? null)
                    ->first();
            }

            $lines = $row['purchase_order_lines'] ?? [];

            foreach ($lines as &$line) {
                if (! empty($line['barcode']) && (empty($line['sku']) || ! isset($line['sku']))) {
                    $product = Product::where('barcode', $line['barcode'])->first();
                    if ($product instanceof Product) {
                        $line['sku'] = $product->sku;
                    }
                }
            }

            if (! $purchaseOrder) {
                if (! $this->isDataValid($row, PurchaseOrderValidator::PURCHASE_ORDER_ACTION_CREATE)) {
                    $this->taskStatus->addErrorMessage("Skipping purchase order number: {$row['purchase_order_number']} due to validation error.");

                    return;
                } // We skip the row since data isn't valid
                $row['purchase_order_lines'] = $lines;
                [$purchaseOrder, $lines] = $this->createPurchaseOrder($row);
            }

            if (! $purchaseOrder->wasRecentlyCreated) {
                // Validate for update.
                if (! $this->isDataValid($row, PurchaseOrderValidator::PURCHASE_ORDER_ACTION_UPDATE, $purchaseOrder)) {
                    $this->taskStatus->addErrorMessage("Skipping sales order number: {$row['purchase_order_number']} due to validation error.");

                    return;
                } // We skip the row since data isn't valid
                [$purchaseOrder, $lines] = $this->updatePurchaseOrder($purchaseOrder, $row);
            }
            // approve
            if ($row['order_status'] !== PurchaseOrder::STATUS_DRAFT && $purchaseOrder->isDraft()) {
                $purchaseOrder->approve();
            }
            // marked as submitted
            if ($row['submission_status'] === PurchaseOrder::SUBMISSION_STATUS_SUBMITTED && $purchaseOrder->submission_status != PurchaseOrder::SUBMISSION_STATUS_FINALIZED) {
                // The purchase order should be marked as submitted in this case but the supplier should not be notified
                if (is_array($res = $purchaseOrder->submit(false))) {
                    $this->validationErrors[$purchaseOrder->purchase_order_number]['submit_error'] = Response::getError(...$res);
                }
            }
            // receipt
            if ($purchaseOrder->isOpen() && collect($lines)->sum('received_quantity') > 0) {
                if (! $this->receivePurchaseOrder($purchaseOrder, $lines, $row)) {
                    $this->taskStatus->addErrorMessage("Skipping receive sales order number: {$row['purchase_order_number']} due to validation error.");
                }
            }
            // invoice
            if (collect($row['purchase_order_lines'])->sum('invoiced_quantity') > 0) {
                if (! $this->invoicePurchaseOrder($purchaseOrder, $lines, $row)) {
                    $this->taskStatus->addErrorMessage("Skipping invoice sales order number: {$row['purchase_order_number']} due to validation error.");
                }
            }
            // Add notes
            if (! empty($row['po_notes'])) {
                if ($purchaseOrder->wasRecentlyCreated) {
                    $purchaseOrder->notes()->create(['note' => $row['po_notes']]);
                } else {
                    $purchaseOrder->notes()->updateOrCreate(['note' => $row['po_notes']]);
                }
            }
        });
    }

    private function isDataValid(array $row, string $action, ?PurchaseOrder $purchaseOrder = null): bool
    {
        $validation = Validator::make(
            $row,
            $this->validator
                ->withAction($action, $purchaseOrder)
                ->rules(),
            $this->validator->messages()
        );

        if ($validation->passes()) {
            $this->validator->withValidator($validation);
        }

        if (! $this->inBackground && $validation->errors()->isNotEmpty()) {
            $this->validationErrors[$row['purchase_order_number']] = $validation->errors()->toArray();
        }

        return $validation->errors()->isEmpty();
    }

    /**
     * Updates the given purchase order with the inputs.
     *
     *
     * @return array [purchaseOrder, lines]
     *
     * @throws \Throwable
     */
    private function updatePurchaseOrder(PurchaseOrder $purchaseOrder, array $inputs): array
    {
        if ($purchaseOrder->inboundShipmentRelation) {
            $this->taskStatus->addErrorMessage(__('messages.purchase_order.purchase_order_is_locked'));

            return [$purchaseOrder, $inputs['purchase_order_lines'] ?? []];
        }

        if (empty($inputs['destination_warehouse_id']) && empty($purchaseOrder->destination_warehouse_id)) {
            $inputs['destination_warehouse_id'] = Helpers::setting(Setting::KEY_PO_DEFAULT_WAREHOUSE);
        }

        $purchaseOrder->update(Arr::except($inputs, 'submission_status'));

        if ($purchaseOrder->order_status != PurchaseOrder::STATUS_CLOSED) {
            $lines = $purchaseOrder->setPurchaseOrderLines(
                array_key_exists('purchase_order_lines', $inputs) ? $inputs['purchase_order_lines'] : false
            );
        }

        return [$purchaseOrder, $lines ?? $inputs['purchase_order_lines'] ?? []];
    }

    /**
     * Creates a new purchase order.
     *
     *
     * @return array [purchaseOrder, lines]
     *
     * @throws \Throwable
     */
    private function createPurchaseOrder(array $inputs): array
    {
        if (empty($inputs['destination_warehouse_id'])) {
            $inputs['destination_warehouse_id'] = Helpers::setting(Setting::KEY_PO_DEFAULT_WAREHOUSE);
        }

        $purchaseOrder = new PurchaseOrder($inputs);
        $purchaseOrder->save();

        $lines = $purchaseOrder->setPurchaseOrderLines($inputs['purchase_order_lines'] ?? null);

        if ($purchaseOrder->destinationWarehouse && $purchaseOrder->destinationWarehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            $purchaseOrder->linkInboundShipmentRelation($inputs['shipment_id']);
        }

        return [$purchaseOrder, $lines];
    }

    /**
     * Receive Purchase Order.
     */
    private function receivePurchaseOrder(PurchaseOrder $purchaseOrder, array $lines, array $row): bool
    {
        $purchaseOrder->load('purchaseOrderLines.purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines');
        // build receipt purchase order request
        $receiptRequestData = ['received_at' => $row['received_at'] ?? now()->toISOString(), 'receipt_lines' => []];
        foreach ($lines as $line) {
            /** @var PurchaseOrderLine $purchaseOrderLine */
            $purchaseOrderLine = isset($line['purchase_order_line_id']) ? $purchaseOrder->purchaseOrderLines->firstWhere('id', $line['purchase_order_line_id']) : PurchaseOrderLine::findByStoreRequest($purchaseOrder->id, $line);
            if ($purchaseOrderLine && $line['received_quantity'] > 0 && $line['received_quantity'] > $purchaseOrderLine->received_quantity) {
                $unreceivedQuantity = $line['received_quantity'] - $purchaseOrderLine->received_quantity;
                $receiptRequestData['receipt_lines'][] = ['purchase_order_line_id' => $purchaseOrderLine->id, 'quantity' => $unreceivedQuantity];
            }
        }

        // already receipted
        if (empty($receiptRequestData['receipt_lines'])) {
            return true;
        }

        try {
            $receiptRequest = ReceiptPurchaseOrderRequest::createFromCustom($receiptRequestData);
        } catch (ValidationException $validationException) {
            $this->validationErrors[$purchaseOrder->purchase_order_number]['receipt_errors'] = $validationException->errors();

            return false;
        } catch (ActionUnavailableTemporaryException $unavailableTemporaryException) {
            $this->validationErrors[$purchaseOrder->purchase_order_number]['receipt_errors'] = $unavailableTemporaryException->getResponseError('purchase_order_number');

            return false;
        }

        (new ShipmentManager())->receiveShipment($receiptRequest->validated());

        return true;
    }

    /**
     * Create Purchase Invoice.
     */
    private function invoicePurchaseOrder(PurchaseOrder $purchaseOrder, array $lines, array $row): bool
    {
        $purchaseOrder->load('purchaseOrderLines.purchaseInvoiceLines');

        $invoiceRequestData = [
            'purchase_invoice_date' => $row['invoice_date'] ?? now()->toISOString(),
            'purchase_invoice_lines' => [],
            'purchase_order_id' => $purchaseOrder->id,
            'supplier_invoice_number' => $row['supplier_invoice_number'] ?? ($purchaseOrder->purchase_order_number.'-INV'),
        ];
        foreach ($lines as $line) {
            /** @var PurchaseOrderLine $purchaseOrderLine */
            $purchaseOrderLine = isset($line['purchase_order_line_id']) ? $purchaseOrder->purchaseOrderLines->firstWhere('id', $line['purchase_order_line_id']) : PurchaseOrderLine::findByStoreRequest($purchaseOrder->id, $line);
            if ($purchaseOrderLine && $line['invoiced_quantity'] > 0 && $line['invoiced_quantity'] > $purchaseOrderLine->invoice_total) {
                $uninvoicedQuantity = $line['invoiced_quantity'] - $purchaseOrderLine->invoice_total;
                $invoiceRequestData['purchase_invoice_lines'][] = ['purchase_order_line_id' => $purchaseOrderLine->id, 'quantity_invoiced' => $uninvoicedQuantity];
            }
        }

        // already invoiced
        if (empty($invoiceRequestData['purchase_invoice_lines'])) {
            return true;
        }

        try {
            $invoiceRequest = StorePurchaseInvoice::createFromCustom($invoiceRequestData);
        } catch (ValidationException $validationException) {
            $this->validationErrors[$purchaseOrder->purchase_order_number]['invoice_errors'] = $validationException->errors();

            return false;
        } catch (ActionUnavailableTemporaryException $unavailableTemporaryException) {
            $this->validationErrors[$purchaseOrder->purchase_order_number]['invoice_errors'] = $unavailableTemporaryException->getResponseError('purchase_order_number');

            return false;
        }

        // Save Purchase Invoice
        $purchaseInvoice = new PurchaseInvoice($invoiceRequest->validated());
        $purchaseInvoice->supplier_id = $purchaseOrder->supplier_id;
        $purchaseOrder->purchaseInvoices()->save($purchaseInvoice);

        // save purchase invoice lines
        $purchaseInvoice->purchaseInvoiceLines()->createMany($invoiceRequestData['purchase_invoice_lines']);

        $purchaseOrder->invoiced($purchaseInvoice->purchase_invoice_date);
        $purchaseOrder->load('purchaseOrderLines.purchaseInvoiceLines');

        return true;
    }

    protected function importByRow(): bool
    {
        // We load the records
        if (! isset($this->records)) {
            $this->loadRecords();
        }

        // We aggregate the purchase order data based on sales order number
        // and prepare them for creation/update.
        $records = collect($this->records);
        $purchaseOrders = [];
        $uniqueOrderNumbers = $records->unique('purchase_order_number')->pluck('purchase_order_number');
        $uniqueOrderNumbers->each(function ($purchaseOrderNumber, $index) use (&$purchaseOrders, $records) {
            // Make the sales order data.
            $rows = array_values($records->where('purchase_order_number', $purchaseOrderNumber)->toArray());
            $row = $rows[0];
            if (! isset($row['order_status'])) {
                $row['order_status'] = PurchaseOrder::STATUS_DRAFT;
            }

            $accepted_date_formats = [
                'd*m*y',
                'd*m*Y',
                'y*m*d',
                'Y*m*d',
            ];

            $date_fields = [
                'purchase_order_date',
                'estimated_delivery_date',
                'other_date',
            ];

            foreach ($date_fields as $date_field) {
                if (! isset($row[$date_field])) {
                    continue;
                }
                foreach ($accepted_date_formats as $format) {
                    if ($dateTime = \DateTime::createFromFormat($format, $row[$date_field])) {
                        $row[$date_field] = ($dateTime->format('m/d/Y'));
                        break;
                    }
                }
            }

            $order = array_filter([
                'purchase_order_number' => $purchaseOrderNumber,
                'submission_format' => $row['submission_format'] ?? null,
                'approval_status' => $row['order_status'] === 'draft' ? PurchaseOrderValidator::APPROVAL_STATUS_PENDING : PurchaseOrderValidator::APPROVAL_STATUS_APPROVED,
                'order_status' => $row['order_status'],
                'submission_status' => $row['submission_status'] ?? PurchaseOrder::SUBMISSION_STATUS_SUBMITTED,
                'purchase_order_date' => $row['purchase_order_date'] ?? null,
                'other_date' => $row['other_date'] ?? null,
                'currency_code' => $row['currency_code'] ?? Currency::default()->code,
                'estimated_delivery_date' => $row['estimated_delivery_date'] ?? null,
                'supplier_id' => $row['supplier_id'] ?? null,
                'supplier_name' => $row['supplier_name'] ?? null,
                'po_notes' => $row['po_notes'] ?? null,
            ]);

            if (! empty($row['shipping_method'])) {
                $shippingMethod = ShippingMethod::with([])->where('full_name', e($row['shipping_method']))->first();
                if ($shippingMethod) {
                    $order['shipping_method_id'] = $shippingMethod->id;
                } else {
                    $order['requested_shipping_method'] = $row['shipping_method'];
                }
            }

            if (! empty($row['destination_warehouse'])) {
                $warehouse = Warehouse::with([])->where('name', e($row['destination_warehouse']))->first();
                $order['destination_warehouse_id'] = $warehouse->id ?? null;
            }

            // Purchase order lines
            $lineKeys = array_filter(array_keys($row), function ($key) {
                return Str::startsWith($key, 'item_');
            });

            $linesInfo = [];
            foreach ($rows as $key => $row) {
                $lineInfo = [];
                foreach ($lineKeys as $lineKey) {
                    $lineInfo[str_replace('item_', '', $lineKey)] = $row[$lineKey];
                    $lineInfo['description'] = $lineInfo['name'] ?? null;
                }

                if (empty($lineInfo['amount'])) {
                    $lineInfo['amount'] = 0;
                }
                if (empty($lineInfo['quantity'])) {
                    $lineInfo['quantity'] = 1;
                }
                if (! empty($row['id'])) {
                    $lineInfo['id'] = $row['id'];
                }
                $linesInfo[] = $lineInfo;
            }

            if (! empty($linesInfo)) {
                $order['purchase_order_lines'] = $linesInfo;
            }

            $purchaseOrders[] = $order;
        });

        // Update the records
        $this->records = $purchaseOrders;

        return parent::importByRow();
    }

    /**
     * Fields for export.
     */
    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'po_number' => Exporter::makeExportableField('purchase_order_number'),
            'purchase_order_date' => Exporter::makeExportableField('purchase_order_date'),
            'other_date' => Exporter::makeExportableField('other_date'),
            'shipping_method' => Exporter::makeExportableField('shipping_method'),
            'estimated_delivery_date' => Exporter::makeExportableField('estimated_delivery_date'),
            'fulfillment_info.shipped' => Exporter::makeExportableField('total_quantity_shipped', false),
            'fulfillment_info.received' => Exporter::makeExportableField('total_quantity_received', false),
            'fulfillment_info.enroute' => Exporter::makeExportableField('total_quantity_enroute', false),
            'supplier_name.name' => Exporter::makeExportableField('supplier_name'),
            'supplier_po_email' => Exporter::makeExportableField('supplier_purchase_order_email'),
            'supplier_email' => Exporter::makeExportableField('supplier_email'),
            'supplier_company' => Exporter::makeExportableField('supplier_company'),
            'supplier_phone' => Exporter::makeExportableField('supplier_phone'),
            'supplier_address' => Exporter::makeExportableField('supplier_address', true, 'Supplier Address Line 1'),
            'supplier_city' => Exporter::makeExportableField('supplier_city'),
            'supplier_province' => Exporter::makeExportableField('supplier_province'),
            'supplier_province_code' => Exporter::makeExportableField('supplier_province_code'),
            'supplier_zip' => Exporter::makeExportableField('supplier_zip'),
            'supplier_country' => Exporter::makeExportableField('supplier_country'),
            'supplier_country_code' => Exporter::makeExportableField('supplier_country_code'),
            'destination_name.name' => Exporter::makeExportableField('destination_warehouse'),
            'destination_po_email' => Exporter::makeExportableField('destination_purchase_order_email'),
            'destination_email' => Exporter::makeExportableField('destination_email'),
            'destination_company' => Exporter::makeExportableField('destination_company'),
            'destination_phone' => Exporter::makeExportableField('destination_phone'),
            'destination_address' => Exporter::makeExportableField('destination_address', true, 'Destination Address Line 1'),
            'destination_city' => Exporter::makeExportableField('destination_city'),
            'destination_province' => Exporter::makeExportableField('destination_province'),
            'destination_province_code' => Exporter::makeExportableField('destination_province_code'),
            'destination_zip' => Exporter::makeExportableField('destination_zip'),
            'destination_country' => Exporter::makeExportableField('destination_country'),
            'destination_country_code' => Exporter::makeExportableField('destination_country_code'),
            'currency_code' => Exporter::makeExportableField('currency_code'),
            'order_status' => Exporter::makeExportableField('order_status', true),
            'submission_status' => Exporter::makeExportableField('submission_status', false),
            'payment_status' => Exporter::makeExportableField('payment_status', false),
            'shipment_status' => Exporter::makeExportableField('shipment_status', false),
            'receipt_status' => Exporter::makeExportableField('receipt_status', false),
            'invoice_status' => Exporter::makeExportableField('invoice_status', false),
            'total' => Exporter::makeExportableField('total', false),
            'product_total' => Exporter::makeExportableField('product_total', false),
            'quantity' => Exporter::makeExportableField('quantity', true),
            'barcode' => Exporter::makeExportableField('barcode', true),
            'sku.sku' => Exporter::makeExportableField('sku', true),
            'supplier_sku' => Exporter::makeExportableField('supplier_sku', true),
            'price' => Exporter::makeExportableField('amount', true),
            'name' => Exporter::makeExportableField('name', true),
            'nominal_code' => Exporter::makeExportableField('nominal_code_name', true),
            'discount' => Exporter::makeExportableField('discount', true),
            'tax' => Exporter::makeExportableField('tax', true),
            'img_url' => Exporter::makeExportableField('img_url', true),
            'received' => Exporter::makeExportableField('received_quantity', true),
            'invoiced' => Exporter::makeExportableField('invoiced_quantity', true),
            'is_variation' => Exporter::makeExportableField('is_variation', true),
            'po_notes' => Exporter::makeExportableField('po_notes', true),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function getRecords()
    {
        return collect(parent::getRecords())->sortBy('purchase_order_number')->values()->toArray();
    }
}
