<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\FileImporter;
use App\Models\Attribute;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductComponent;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Repositories\WarehouseRepository;
use App\Response;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class ProductDataImporter.
 */
class ProductDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'sku',
    ];

    protected $deferredRows = [];

    /**
     * @var string
     */
    protected $idColumn = 'sku';

    /**
     * ProductDataImporter constructor.
     */
    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);
    }

    /**
     * Gets exportable fields.
     */
    public static function getExportableFields(): array
    {
        $fields = [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'sku' => Exporter::makeExportableField('sku', true, 'SKU'),
            'image' => Exporter::makeExportableField('image_url'),
            'other_images' => Exporter::makeExportableField('other_images'),
            'name' => Exporter::makeExportableField('name'),
            'brand.name' => Exporter::makeExportableField('brand', true, 'Brand Name'),
            'barcode' => Exporter::makeExportableField('barcode'),
            'mpn' => Exporter::makeExportableField('mpn', true, 'MPN'),
            'type' => Exporter::makeExportableField('type'),
            'unit_cost' => Exporter::makeExportableField('unit_cost'),
            'parent_id' => Exporter::makeExportableField('parent_id', true, 'Product ID'),
            'has_fba' => Exporter::makeExportableField('has_fba'),
            'is_variation' => Exporter::makeExportableField('is_variation', false),
            'weight' => Exporter::makeExportableField('weight'),
            'weight_unit' => Exporter::makeExportableField('weight_unit'),
            'length' => Exporter::makeExportableField('length'),
            'width' => Exporter::makeExportableField('width'),
            'height' => Exporter::makeExportableField('height'),
            'dimension_unit' => Exporter::makeExportableField('dimension_unit'),
            'fba_prep_instructions' => Exporter::makeExportableField('fba_prep_instructions'),
            'case_quantity' => Exporter::makeExportableField('case_quantity'),
            'case_length' => Exporter::makeExportableField('case_length'),
            'case_width' => Exporter::makeExportableField('case_width'),
            'case_dimensions_unit' => Exporter::makeExportableField('case_dimensions_unit'),
            'average_cost.value' => Exporter::makeExportableField('average_cost', false),
            'average_cost.currency' => Exporter::makeExportableField('average_cost_currency', false),
            'default_supplier.name' => Exporter::makeExportableField('default_supplier_name'),
            'default_supplier_sku' => Exporter::makeExportableField('default_supplier_sku'),
            'default_supplier_leadtime' => Exporter::makeExportableField('default_supplier_leadtime'),
            'inventory_total' => Exporter::makeExportableField('inventory_total', false),
            'inventory_available' => Exporter::makeExportableField('inventory_available', false),
            'inventory_reserved' => Exporter::makeExportableField('inventory_reserved', false),
            'inventory_incoming' => Exporter::makeExportableField('inventory_incoming', false),
            'inventory_in_transit' => Exporter::makeExportableField('inventory_in_transit', false),
            'inventory_stock_value' => Exporter::makeExportableField('inventory_stock_value', false),
            'daily_average_consumption' => Exporter::makeExportableField('daily_average_consumption', false),
            'category_main.name' => Exporter::makeExportableField('category_main.name', true, 'Main Category Name'),
            'category_main_path' => Exporter::makeExportableField('category_main_path', true, 'Main Category Path'),
            'category_others' => Exporter::makeExportableField('category_others', true, 'Other Categories'),
            'category_others_path' => Exporter::makeExportableField('category_others_path', true, 'Other Category Paths'),
            'tags' => Exporter::makeExportableField('tags', true, 'Tags'),
            'component_of_sku' => Exporter::makeExportableField('component_of_sku', true, 'component_of_sku'),
            'component_of_id' => Exporter::makeExportableField('component_of_id', true, 'component_of_id'),
            'component_skus' => Exporter::makeExportableField('component_skus', true, 'component_skus'),
            'component_ids' => Exporter::makeExportableField('component_ids', true, 'component_ids'),
            'suppliers' => Exporter::makeExportableField('suppliers', true, 'suppliers'),
        ];

        // Add in attributes
        Attribute::all()->each(function (Attribute $attribute) use (&$fields) {
            $fields['attributes.'.$attribute->name.'.value'] = Exporter::makeExportableField(
                'attributes.'.$attribute->name,
                true,
                'Attributes:'.ucfirst($attribute->name)
            );
        });

        // Pricing tiers
        $tiers = ProductPricingTier::all();
        foreach ($tiers as $pricingTier) {
            $value = ('price.'.$pricingTier->name.'.value');
            $fields[$value] = Exporter::makeExportableField($value);
            $currency = ('price.'.$pricingTier->name.'.currency');
            $fields[$currency] = Exporter::makeExportableField($currency, false);
        }

        // Supplier Pricing tiers
        $supplierPricingTiers = SupplierPricingTier::all();
        foreach ($supplierPricingTiers as $pricingTier) {
            $value = 'supplier_pricing.'.$pricingTier->name.'.value';
            $fields[$value] = Exporter::makeExportableField($value);
            $currency = ('supplier_pricing.'.$pricingTier->name.'.currency');
            $fields[$currency] = Exporter::makeExportableField($currency, false);
        }

        // Warehouse inventory
        $warehouses = Warehouse::with([])
            ->where('type', '!=', Warehouse::TYPE_SUPPLIER)
            ->where('type', '!=', Warehouse::TYPE_AMAZON_FBA)
            ->get(['id', 'name']);

        foreach ($warehouses as $warehouse) {
            $inventory = 'inventory.'.$warehouse->name;
            $fields[$inventory] = Exporter::makeExportableField($inventory, false);
            $inventoryAvailableWarehouses = 'inventory_available_warehouses.'.$warehouse->name;
            $fields[$inventoryAvailableWarehouses] = Exporter::makeExportableField($inventoryAvailableWarehouses, false);
        }

        return array_merge($fields, [
            'archived_at' => Exporter::makeExportableField('archived_at', false),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ]);
    }

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            $created = self::createProduct($row, $this);
            $keysToCheck = ['component_of_sku', 'component_skus', 'component_of_id', 'component_ids'];

            if (array_intersect_key(array_flip($keysToCheck), $row)) {
                $this->deferredRows[] = $row;
            }

            return $created;
        });
    }

    /**
     * @param  null  $importer
     *
     * @throws BindingResolutionException
     * @throws FileNotFoundException
     */
    public static function createProduct(array $row, ?FileImporter $importer = null): ?Product
    {
        // We create the given product or update it if possible
        // We attempt to find by id or sku
        if (! empty($row['id'])) {
            $product = Product::with([])->find($row['id']);
            if (! $product) {
                if ($importer) {
                    $importer->validationErrors[$row['sku'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'sku']));
                    $importer->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");
                }

                return null;
            }
        } else {
            $product = Product::with([])
                ->where('sku', $row['sku'])
                ->first();
        }

        if (! $product) {
            $product = new Product;
            if (! isset($row['type'])) {
                $product->type = Product::TYPE_STANDARD;
            }
        }

        // Sku cannot be blank
        if (empty($row['sku'])) {
            $importer?->taskStatus->addErrorMessage('Skipping product with empty sku.');

            return null;
        }

        // Use sku as name if not provided
        if ($product->wasRecentlyCreated && empty($row['name'])) {
            $row['name'] = $row['sku'];
        }

        $product = $product->fill($row);
        $defaultDimension = Setting::getValueByKey(Setting::KEY_DEFAULT_DIMENSION_UNIT);
        $defaultWeight = Setting::getValueByKey(Setting::KEY_DEFAULT_WEIGHT_UNIT);

        if (empty($row['dimension_unit'])) {
            $product->dimension_unit = $defaultDimension;
        }
        if (empty($row['weight_unit'])) {
            $product->weight_unit = $defaultWeight;
        }
        if (empty($row['case_dimension_unit'])) {
            $product->case_dimension_unit = $defaultDimension;
        }
        if (empty($row['case_weight_unit'])) {
            $product->case_weight_unit = $defaultWeight;
        }

        if (! empty($row['unit_cost'])) {
            $product->unit_cost = trim($row['unit_cost'], '$');
        }

        $product->save();

        // Set main category
        if (! empty($row['category_main'])) {
            $category = ProductCategory::with([])->firstOrCreate(['name' => $row['category_main']]);
            $mainCategory = [
                'category_id' => $category->id,
                'is_primary' => true,
            ];
            if (! empty($row['category_main_path'])) {
                $mainCategory['category_path'] = $row['category_main_path'];
            }

            $product->setCategories([
                $mainCategory,
            ]);
        }

        // Set other fields if available
        if (! empty($row['brand'])) {
            $product->setBrandName($row['brand'], true);
        }

        if (! empty($row['description'])) {
            $product->setProductAttributes([
                'name' => 'description',
                'value' => $row['description'],
            ], false);
        }

        $images = [];
        if (! empty($row['image_url']) || ! empty($row['image'])) {
            $images[] = [
                'url' => $row['image_url'] ?? $row['image'],
                'download' => $row['image_download'] ?? false,
                'is_primary' => true,
                'operation' => Helpers::OPERATION_UPDATE_CREATE,
            ];
        }

        if (! empty($row['other_images'])) {
            $otherImages = explode(',', $row['other_images']);
            foreach ($otherImages as $url) {
                $url = trim($url);
                $urlExists = $product->images()->where('url', $url)->exists();
                if ($url && ! $urlExists && ! Str::startsWith('/storage/images', $url)) {
                    $images[] = [
                        'url' => $url,
                        'download' => false,
                        'is_primary' => false,
                        'operation' => Helpers::OPERATION_UPDATE_CREATE,
                    ];
                }
            }
        }

        if (! empty($images)) {
            $product->addImages($images);
        }

        if (isset($row['default_supplier']) && empty($row['default_supplier_name'])) {
            $row['default_supplier_name'] = $row['default_supplier'];
        }

        if (! empty($row['default_supplier_sku'])) {
            $supplier_product = SupplierProduct::with([])
                ->where('product_id', $product->id)
                ->where('is_default', true)
                ->first();

            if ($supplier_product) {
                $supplier_product->update([
                    'supplier_sku' => $row['default_supplier_sku'],
                ]);
            }
        }

        if (! empty($row['default_supplier_name'])) {
            self::setDefaultSupplierProducts($product, $row);
        }

        if (array_key_exists('suppliers', $row)) {
            $product->supplierProducts()->delete();
            if (! empty($row['suppliers'])) {
                $suppliersData = json_decode(self::fixIncorrectJson($row['suppliers']), true);
                foreach ($suppliersData as $supplierData) {
                    self::setSupplierProducts($product, $supplierData);
                    if (isset($supplierData['supplier_pricing'])) {
                        self::syncSupplierPricings($product, $supplierData['supplier_pricing'], $supplierData['name']);
                    }
                }
            }
        }

        // Set pricing
        self::setProductPricing($product, $row);

        self::setSupplierPricing($product, $row);
        // Set Attributes
        self::setProductAttributes($product, $row);

        if (! empty($row['tags'])) {
            $product->attachTags(explode('|', $row['tags']));
        }

        return $product;
    }

    /**
     * Sets product attributes.
     */
    public static function setProductAttributes(Product $product, $row)
    {
        $attributes = [];
        foreach (collect($row)->filter(fn ($a, $key) => Str::startsWith($key, 'attributes.')) as $key => $value) {
            $attributes[str_replace('attributes.', '', $key)] = $value;
        }

        Attribute::query()->whereIn('name', array_keys($attributes))->each(function (Attribute $attribute) use ($product, $attributes) {
            $payload = [
                'id' => $attribute->id,
                'value' => $attributes[$attribute->name],
            ];
            $product->setProductAttributes([$payload], false);
        });
    }

    /**
     * @throws BindingResolutionException
     */
    public static function setDefaultSupplierProducts(Product $product, array $row)
    {
        // If the supplier doesn't exist,
        // we create the supplier on the fly.
        $supplier = Supplier::with([])->firstOrCreate(['name' => $row['default_supplier_name']]);
        if ($supplier->wasRecentlyCreated) {
            // Add default address
            $supplier->updateOrCreateAddress([
                'label' => Supplier::OFFICE_ADDRESS_LABEL,
                'name' => $supplier->name,
            ]);

            // Attach pricing tier
            $tiers = SupplierPricingTier::with([])->where('is_default', true)->pluck('id')->toArray();
            $supplier->attachPricingTiers($tiers);

            // Create default warehouse if not available
            if (! $supplier->defaultWarehouse) {
                /** @var WarehouseRepository $warehouses */
                $warehouses = app()->make(WarehouseRepository::class);
                $warehouses->createDefaultWarehouseForSupplier($supplier);
            }
        }

        // Set or update the default supplier

        $products = [
            'supplier_id' => $supplier->id,
            'supplier_sku' => $row['default_supplier_sku'] ?? null,
            'is_default' => true,
            'leadtime' => $row['default_supplier_leadtime'] ?? null,
            'minimum_order_quantity' => $row['minimum_order_quantity'] ?? null,
        ];
        $product->setSupplierProducts([$products], false);
    }

    public static function fixIncorrectJson(string $jsonString): string
    {
        // Add double quotes around keys
        $jsonString = preg_replace('/(?<!\\\\)([a-zA-Z_]\w*+):/', '"$1":', $jsonString);

        // Add double quotes around non-numeric values except for boolean and null
        $jsonString = preg_replace('/:\s*(?!"|\[|\{|true|false|null)([^,\}\]]+)/', ':"$1"', $jsonString);

        return $jsonString;
    }

    public static function setProductPricing(Product $product, $row)
    {
        $productPricing = collect(array_keys($row))->filter(function ($key) {
            return Str::startsWith($key, 'price.');
        });

        if ($productPricing->isEmpty()) {
            return;
        }

        $productPricing = $productPricing->map(function ($field) {
            return explode('.', $field)[1]; // Pricing name is the middle value
        })->unique()->toArray();

        $payload = [];
        foreach ($productPricing as $tierName) {
            $pricingTier = ProductPricingTier::with([])
                ->where('name', $tierName)
                ->first();

            if (! $pricingTier) {
                continue;
            }

            $price = $row['price.'.$tierName.'.value'];

            $payload[] = [
                'product_pricing_tier_id' => $pricingTier->id,
                'price' => $price ?? 0,
                'operation' => Helpers::OPERATION_UPDATE_CREATE,
            ];
        }

        $product->setPricing($payload, false);
    }

    public static function setSupplierPricing(Product $product, array $row)
    {
        $supplierPricing = collect(array_keys($row))->filter(function ($key) {
            return Str::startsWith($key, 'supplier_pricing.');
        });

        if ($supplierPricing->isEmpty()) {
            return;
        }

        $supplierPricing = $supplierPricing->map(function ($field) {
            return explode('.', $field)[1]; // Pricing name is the middle value
        })->unique()->toArray();

        $payload = [];
        foreach ($supplierPricing as $tierName) {
            $pricingTier = SupplierPricingTier::with([])
                ->where('name', $tierName)
                ->firstOrFail();

            $supplier = $product->suppliers()->where('is_default', true)->first();
            if (! $supplier) {
                continue;
            }

            $price = $row['supplier_pricing.'.$tierName.'.value'];

            $payload[] = [
                'supplier_id' => $supplier->id,
                'pricing' => [
                    [
                        'supplier_pricing_tier_id' => $pricingTier->id,
                        'supplier_pricing_tier_name' => $pricingTier->name,
                        'price' => $price ?? 0,
                        'operation' => Helpers::OPERATION_UPDATE_CREATE,
                    ],
                ],
            ];
        }

        $product->setSupplierProducts($payload, false);
    }

    protected function finalizeImport()
    {
        return $this->establishRelationships();
    }

    protected function establishRelationships()
    {
        try {
            foreach ($this->deferredRows as $row) {
                $this->establishRelationship($row, $this);
            }

            return true;
        } catch (\Exception $exception) {
            return false;
        }
    }

    protected function establishRelationship(array $row, ?FileImporter $importer = null)
    {
        $product = Product::where('sku', $row['sku'])->firstOrFail();

        if (array_key_exists('component_skus', $row)) {
            $this->attachComponents($product, $row['component_skus'], 'sku', $importer);
        }

        if (array_key_exists('component_ids', $row)) {
            $this->attachComponents($product, $row['component_ids'], 'id', $importer);
        }

        if (array_key_exists('component_of_sku', $row)) {
            $this->associateToParents($product, $row['component_of_sku'], 'sku', $importer);
        }

        if (array_key_exists('component_of_id', $row)) {
            $this->associateToParents($product, $row['component_of_id'], 'id', $importer);
        }
    }

    private function attachComponents($product, $componentIdentifiers, $field, $importer)
    {
        $componentIdentifiersArray = $this->convertSkusToArray($componentIdentifiers);
        $components = Product::whereIn($field, array_keys($componentIdentifiersArray))->get();

        ProductComponent::where('parent_product_id', $product->id)->whereNotIn('component_product_id', $components->pluck('id'))->delete();
        if ($components->isNotEmpty()) {
            $syncData = $components->mapWithKeys(function ($component) use ($componentIdentifiersArray, $field) {
                $sku = $component->{$field};
                $quantity = $componentIdentifiersArray[$sku] ?? 1;

                return [$component->id => ['quantity' => $quantity]];
            });
            $product->components()->sync($syncData);
        }
    }

    public function convertSkusToArray($componentSkus)
    {
        // Initialize the resulting array
        $resultArray = [];

        // Split the string by comma
        $skus = explode(',', $componentSkus);

        // Loop through each SKU
        foreach ($skus as $sku) {
            // Split the SKU by the arrow symbol (=>)
            $parts = preg_split('/->|→/', $sku);
            // Extract the key and value
            $key = trim($parts[0]);
            $value = (isset($parts[1])) ? intval(trim($parts[1])) : 1;

            // Add the key-value pair to the result array
            $resultArray[$key] = $value;
        }

        return $resultArray;
    }

    private function associateToParents($product, $parentIdentifiers, $field, $importer)
    {
        $parentIdentifiersArray = $this->convertSkusToArray($parentIdentifiers);
        $parents = Product::whereIn($field, array_keys($parentIdentifiersArray))->get();
        ProductComponent::where('component_product_id', $product->id)->whereNotIn('parent_product_id', $parents->pluck('id'))->delete();

        if ($parents->isNotEmpty()) {
            $syncData = $parents->mapWithKeys(function ($parent) use ($parentIdentifiersArray, $field) {
                $sku = $parent->{$field};
                $quantity = $parentIdentifiersArray[$sku] ?? 1;

                return [$parent->id => ['quantity' => $quantity]];
            });
            $product->parentProducts()->sync($syncData);
        }
    }

    public static function setSupplierProducts(Product $product, array $row)
    {
        // If the supplier doesn't exist,
        // we create the supplier on the fly.
        $supplier = Supplier::with([])->firstOrCreate(['name' => $row['name']]);

        if ($supplier->wasRecentlyCreated) {
            // Add default address
            $supplier->updateOrCreateAddress([
                'label' => Supplier::OFFICE_ADDRESS_LABEL,
                'name' => $supplier->name,
            ]);

            // Attach pricing tier
            $tiers = SupplierPricingTier::with([])->where('is_default', true)->pluck('id')->toArray();
            $supplier->attachPricingTiers($tiers);

            // Create default warehouse if not available
            if (! $supplier->defaultWarehouse) {
                /** @var WarehouseRepository $warehouses */
                $warehouses = app()->make(WarehouseRepository::class);
                $warehouses->createDefaultWarehouseForSupplier($supplier);
            }
        }

        // Set or update the default supplier

        $products = [
            'supplier_id' => $supplier->id,
            'supplier_sku' => $row['supplier_sku'] ?? null,
            'is_default' => (bool) ($row['is_default'] ?? false),
            'leadtime' => $row['leadtime'] ?? null,
        ];
        $product->setSupplierProducts([$products], false);
    }

    public static function syncSupplierPricings(Product $product, array $rows, $supplierName)
    {
        if (empty($rows)) {
            return;
        }

        $supplier = $product->suppliers()->where('name', $supplierName)->first();
        if (! $supplier) {
            return;
        }

        $payload = [];
        foreach ($rows as $row) {
            if (empty($row['name']) || empty($row['currency_code'])) {
                return;
            }
            $attributes = [
                'name' => $row['name'],
                'currency_code' => $row['currency_code'],
            ];

            $pricingTier = SupplierPricingTier::firstOrCreate($attributes);

            $payload[] = [
                'supplier_id' => $supplier->id,
                'pricing' => [
                    [
                        'supplier_pricing_tier_id' => $pricingTier->id,
                        'supplier_pricing_tier_name' => $pricingTier->name,
                        'price' => $row['price'] ?? 0,
                        'operation' => Helpers::OPERATION_UPDATE_CREATE,
                    ],
                ],
            ];
        }

        $product->setSupplierProducts($payload, false);
    }
}
