<?php

namespace App\Importers\DataImporters;

use App\Data\CreateInitialStockTakeData;
use App\Data\CreateInitialStockTakeItemData;
use App\Data\StockTakeItemData;
use App\Data\UpdateStockTakeData;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Helpers;
use App\Importers\DataImporter;
use App\Models\Product;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Services\StockTake\StockTakeManager;
use Exception;
use Illuminate\Support\Collection;
use Throwable;

class InitialInventoryDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'sku',
    ];

    private StockTakeManager $stockTakeManager;

    private Warehouse $warehouse;
    private Collection $items;

    public function __construct($taskId, string $filePath)
    {
        parent::__construct($taskId, $filePath);
        $this->stockTakeManager = app(StockTakeManager::class);
        $this->items = collect();
    }

    /**
     * @throws Exception
     */
    protected function beforeImport(): void
    {
        if (!isset($this->meta['warehouse_id'])) {
            throw new Exception('Warehouse must pe provided for initial inventory import');
        }

        $this->warehouse = Warehouse::with('initialCountStockTake')->findOrFail($this->meta['warehouse_id']);
    }

    /**
     * @throws Exception
     */
    protected function importRow(array $row): void
    {
        $product = null;
        if (!empty($row['sku'])) {
            $product = Product::where('sku', $row['sku'])->first();
        } elseif (!empty($row['barcode'])) {
            $product = Product::where('barcode', $row['barcode'])->first();
        }

        if (!$product) {
            return;
        }

        $this->items->add(CreateInitialStockTakeItemData::from([
            'product_id' => $product->id,
            'qty_counted' => (float) $row['quantity'],
            'unit_cost' => (float) $row['unit_cost'],
        ]));
    }

    /**
     * @throws Throwable
     */
    protected function finalizeImport(): void
    {
        if ($stockTake = $this->warehouse->initialCountStockTake) {
            $stockTakeData = UpdateStockTakeData::from([
                'items' => StockTakeItemData::collection($this->items),
            ]);
            $this->stockTakeManager->modifyStockTake($stockTake, $stockTakeData);
        } else {
            $stockTakeData = CreateInitialStockTakeData::from([
                'date_count' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE),
                'warehouse_id' => $this->warehouse->id,
                'items' => CreateInitialStockTakeItemData::collection($this->items),
            ]);

            $this->stockTakeManager->createInitialStockTake($stockTakeData);
        }
    }

    /**
     * Fields for export.
     */
    public static function getExportableFields(): array
    {
        return [
            'sku' => Exporter::makeExportableField('sku', true, 'ID'),
            'quantity' => Exporter::makeExportableField('quantity', true),
            'unit_cost' => Exporter::makeExportableField('unit_cost', true)
        ];
    }
}
