<?php

namespace App\Importers\DataImporters;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Importers\DataImporter;
use App\Models\Attribute;
use App\Response;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

/**
 * Class AttributeDataImporter.
 */
class AttributeDataImporter extends DataImporter
{
    /**
     * @var string[]
     */
    protected $requiredColumns = [
        'name',
        'type',
    ];

    /**
     * @return mixed|void
     */
    protected function importRow(array $row)
    {
        DB::transaction(function () use ($row) {
            // We create the given attribute or update it if possible
            // We attempt to find by id or name and type
            if (! empty($row['id'])) {
                $attribute = Attribute::with([])->find($row['id']);
                if (! $attribute) {
                    $this->validationErrors[$row['name'] ?? $row['id']] = Response::getError(__('messages.import_export.id_not_exists', ['id' => $row['id']]), Response::CODE_NOT_FOUND, 'id', Arr::only($row, ['id', 'name']));
                    $this->taskStatus->addErrorMessage("Skipping id: The id {$row['id']} doesn't exist in SKU");

                    return;
                }
            } else {
                $attribute = Attribute::with([])->where('name', $row['name'])
                    ->where('type', $row['type'])
                    ->firstOrNew();
            }

            $attribute->name = $row['name'];
            $attribute->type = $row['type'];
            $attribute->attribute_group_id = $attribute['attribute_group_id'] ?? null;
            $attribute->display = $this->makeDisplayOptions($attribute, $row);
            $attribute->validation = $this->makeValidation($attribute, $row);

            $attribute->save();
        });
    }

    private function makeDisplayOptions(Attribute $attribute, array $row): array
    {
        $displayOptions = [];

        if (! $attribute->wasRecentlyCreated) {
            $displayOptions = $attribute->display ?? [];
        }

        if ($this->hasDisplayOptions($row)) {
            $displayOptions['has_options'] = $row['has_options'] ?? false;
            $displayOptions['type'] = ucfirst($row['options_type'] ?? Attribute::TYPE_STRING ?? '');
        }

        $displayOptions['sort_order'] = ! empty($row['sort_order']) ? $row['sort_order'] : 0;

        if (! empty($row['is_link'])) {
            $displayOptions['is_link'] = boolval($row['is_link']);
        }

        if (! empty($row['options_values'])) {
            $values = explode(',', $row['options_values']);
            foreach ($values as $key => $value) {
                $attribute->values()->updateOrCreate(['value' => $value], [
                    'value' => $value,
                    'sort_order' => $key,
                ]);
            }
        }

        return $displayOptions;
    }

    private function makeValidation(Attribute $attribute, array $row): array
    {
        $validation = [];

        if (! $attribute->wasRecentlyCreated) {
            $validation = $attribute->validation ?? [];
        }

        if (isset($row['validation.character_limit']) && ! empty($row['validation.character_limit'])) {
            $validation['limit'] = $row['validation.character_limit'];
        }

        if (isset($row['validation.min']) && is_numeric($row['validation.min'])) {
            $validation['min'] = $row['validation.min'];
        }

        if (isset($row['validation.max']) && is_numeric($row['validation.max'])) {
            $validation['max'] = $row['validation.max'];
        }

        if (isset($row['validation.precision'])) {
            $validation['precision'] = $row['validation.precision'];
        }

        return $validation;
    }

    private function hasDisplayOptions(array $row): bool
    {
        return isset($row['has_options']) && $row['has_options'] == true;
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'name' => Exporter::makeExportableField('name'),
            'type' => Exporter::makeExportableField('type', true, 'Attribute Type'),
            'display_options.has_options' => Exporter::makeExportableField('has_options', true, 'Has Display Options'),
            'display_options.type' => Exporter::makeExportableField('options_type', true, 'Display Option Type'),
            'attribute_group_id' => Exporter::makeExportableField('attribute_group_id', true, 'Attribute Group ID'),
            'validation.character_limit' => Exporter::makeExportableField('validation.character_limit', true, 'Validation: Character Limit'),
            'validation.precision' => Exporter::makeExportableField('validation.precision', true, 'Validation: Precision'),
            'validation.min' => Exporter::makeExportableField('validation.min', true, 'Validation: Min'),
            'validation.max' => Exporter::makeExportableField('validation.max', true, 'Validation: Max'),
            'is_link' => Exporter::makeExportableField('is_link'),
            'sort_order' => Exporter::makeExportableField('sort_order'),
            'options_values' => Exporter::makeExportableField('options_values'),
        ];
    }
}
