<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\Product;
use App\Models\SalesOrderLine;
use Illuminate\Http\Request;

/**
 * Class SalesOrderLineResource.
 *
 *
 * @mixin SalesOrderLine
 */
class SalesOrderLineResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        /** @var Product $product */
        $product = $this->resource->relationLoaded('product') ? $this->product : null;

        return [
            'sales_order_line_id' => $this->id,
            'sales_channel_line_id' => $this->inclusive('sales_channel_line_id', $this->sales_channel_line_id),
            'unit_cost' => $this->inclusive('unit_cost', fn () => $this->whenLoaded('salesOrderLineFinancial', $this->salesOrderLineFinancial?->cogs ?? 0)),
            'tax_rate' => $this->inclusive('tax_rate', $this->tax_rate),
            'tax_rate_id' => $this->inclusive('tax_rate_id', $this->tax_rate_id),
            'item_tax_value' => $this->inclusive('item_tax_value', $this->tax_allocation),
            'fifo_layer_id' => $this->inclusive('fifo_layer_id', $this->fifo_layer_id),
            'item_quantity' => $this->inclusive('item_quantity', $this->quantity),
            'item_canceled_quantity' => $this->inclusive('item_canceled_quantity', $this->canceled_quantity),
            'item_reserved_quantity' => $this->inclusive('item_reserved_quantity', fn () => $this->whenLoaded('inventoryMovements', $this->reserved_quantity)),
            'item_unreserved_quantity' => $this->inclusive('item_unreserved_quantity', fn () => $this->whenLoaded('inventoryMovements', $this->unreserved_quantity)),
            'item_externally_fulfilled_quantity' => $this->inclusive('item_externally_fulfilled_quantity', $this->externally_fulfilled_quantity),
            'item_dropshipped_quantity' => $this->inclusive('item_dropshipped_quantity', fn () => $this->whenLoaded('salesOrder.purchaseOrders', $this->dropshipped_quantity)),
            'item_sku' => $this->inclusive('item_sku', [
                'id' => $this->product_id,
                'sku' => $product->sku ?? null,
            ]),
            'item_bundle_sku' => $this->inclusive('item_bundle_sku', fn () => $this->whenLoaded('bundle', $this->bundle?->sku)),
            'item_barcode' => $this->inclusive('item_barcode', $product ? $product->barcode : null),
            'item_brand' => $this->inclusive('item_brand', [
                'id' => $product?->brand_id,
                'name' => $product?->brand?->name,
            ]),
            'bundle' => $this->whenLoaded('product', function () {
                return $this->whenLoaded('bundle', function () {
                    return $this->bundle ? [
                        'id' => $this->bundle->id,
                        'name' => $this->bundle->name,
                        'sku' => $this->bundle->sku,
                        'barcode' => $this->bundle->barcode,
                    ] : null;
                });
            }),
            'item_price' => $this->inclusive('item_price', $this->amount),
            'item_price_with_tax' => $this->inclusive('item_price_with_tax', ($this->amount + $this->tax_total)),
            'item_name' => $this->inclusive('item_name', $this->description),
            'item_nominal_code_id' => $this->inclusive('item_nominal_code_id', $this->nominal_code_id),
            'item_nominal_code' => $this->inclusive('item_nominal_code', $this->nominalCode->name ?? null),
            'item_nominal_code_code' => $this->inclusive('item_nominal_code', $this->nominalCode->code ?? null),
            'item_discount'          => $this->inclusive('item_discount', $this->discount),
            'item_has_backorder'     => $this->inclusive('item_has_backorder', $this->has_active_backorder),
            'is_product'             => $this->inclusive('is_product', $this->is_product),
            'is_bundle'              => $this->inclusive('is_bundle', $product ? $this->product->type == Product::TYPE_BUNDLE : null),
            'is_variation'           => $this->inclusive('is_variation', $this->product ? $this->product->is_variation : false),
            'matrix_product'         => $this->when($this->product !== null && $this->product->is_variation, function () {
                return $this->inclusive('matrix_product', [
                    'id' => $this->product->parent->id,
                    'sku' => $this->product->parent->sku,
                ]);
            }),
            'credited_quantity' => $this->inclusive('credited_quantity', $this->credited_quantity),
            'warehouse_routing_method' => $this->inclusive('warehouse_routing_method', $this->warehouse_routing_method),
            'warehouse' => $this->whenLoaded('warehouse', function () {
                return $this->warehouse?->only('id', 'name', 'supplier_id', 'supplier_name', 'type', 'order_fulfillment');
            }, $this->warehouse_id),
            'is_fulfilled' => $this->inclusive('is_fulfilled', $this->when($this->resource->relationLoaded('salesOrderFulfillmentLines'), function () {
                return $this->fulfilled;
            })),
            'no_audit_trail' => $this->inclusive('no_audit_trail', $this->no_audit_trail),
            'unfulfilled_quantity' => $this->inclusive('unfulfilled_quantity', $this->when($this->resource->relationLoaded('salesOrderFulfillmentLines'), function () {
                return $this->unfulfilled_quantity;
            })),
            'returnable_quantity' => $this->inclusive('returnable_quantity', $this->when($this->resource->relationLoaded('salesCreditLines') && $this->resource->relationLoaded('salesOrderFulfillmentLines'), function () {
                return $this->unreturned_quantity;
            })),
            'backorder_queue_id' => $this->inclusive('backorder_queue_id', $this->whenLoaded('backorderQueue', function () {
                return $this->backorderQueue?->id;
            })),
            'backordered_quantity' => $this->inclusive('backordered_quantity', $this->whenLoaded('backorderQueue', function () {
                return $this->backorderQueue ? [
                    'total_backordered' => $this->backorderQueue->backordered_quantity,
                    'total_released' => $this->backorderQueue->released_quantity,
                    'covered_quantity' => $this->backorderQueue->covered_quantity,
                    'unallocated_quantity' => $this->backorderQueue->unallocated_backorder_quantity,
                    'scheduled' => $this->backorderQueue->getSchedule(),
                ] : null;
            })),
            'listing' => $this->inclusive('listing', fn () => $this->getListing()),
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,

            $this->mergeWhen(! is_null($product), function () use ($product) {
                return [
                    'is_taxable' => $product->is_taxable,
                    'default_tax_rate_id' => $product->default_tax_rate_id,
                ];
            }),

            $this->mergeWhen($product && $product->relationLoaded('primaryImage'), function () use ($product) {
                return [
                    'item_img_url' => $this->inclusive('item_img_url', $product->image ?? null),
                ];
            }),

            $this->mergeWhen($product && $product->relationLoaded('totalInventory'), function () use ($product) {
                return [
                    'item_stock' => $this->inclusive('item_stock', $product->totalInventory->inventory_available ?? 0),
                ];
            }),

            $this->mergeWhen($product && $product->relationLoaded('productInventory'), function () use ($product) {
                return ['item_inventory' => SalesOrderLineInventoryResource::make($product)];
            }),
        ];
    }
}
