<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Models\Currency;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentLine;
use Illuminate\Http\Request;

/**
 * Class PurchaseOrderResource.
 *
 *
 * @mixin PurchaseOrder
 */
class PurchaseOrderResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        return [
            //-----------------------------------------------------
            // Basic Info
            //-----------------------------------------------------

            'po_number' => $this->inclusive('po_number', $this->purchase_order_number),
            'purchase order number' => $this->inclusive('po_number', $this->purchase_order_number), // User friendly name
            'id' => $this->id,
            /** @see https://siberventures.atlassian.net/browse/SKU-1317 */
            'purchase_order_date' => $this->inclusive('purchase_order_date', $this->purchase_order_date),
            'other_date' => $this->inclusive('other_date', $this->other_date ?? null),
            'payment_term_id' => $this->inclusive('payment_term_id', $this->payment_term_id),
            'payment_term' => $this->inclusive('payment_term', $this->whenLoaded('paymentTerm', function () {
                return $this->paymentTerm->name;
            })),
            'incoterm_id' => $this->inclusive('incoterm_id', $this->incoterm_id),
            'incoterm' => $this->inclusive('incoterm', $this->whenLoaded('incoterm', function () {
                return $this->incoterm->description;
            })),
            'submission_format' => $this->inclusive('submission_format', $this->submission_format),
            'currency' => $this->inclusive('currency', $this->currency->code),
            'currency_rate' => $this->inclusive('currency_rate', $this->currency_rate),
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),
            'archived_at' => $this->inclusive('archived_at', $this->archived_at),
            'is_dropship' => isset($this->sales_order_id) && ! isset($this->destination_warehouse_id),
            'tracking_number' => $this->inclusive('tracking_number', $this->tracking_number),

            /*
            |--------------------------------------------------------------------------
            | Inbound Shipment
            |--------------------------------------------------------------------------
            */
            'inbound_shipment' => $this->whenLoaded('inboundNewShipmentRelation', function () {
                $address = $this->inboundNewShipmentRelation->destination['address'];
                return [
                    'id' => $this->inboundNewShipmentRelation->id,
                    'ShipmentId' => $this->inboundNewShipmentRelation->shipmentConfirmationId,
                    'DestinationFulfillmentCenterId' => $this->inboundNewShipmentRelation->destinationWarehouse,
                    'destinationAddress' => [
                        'company' => $address['name'],
                        'address1' => $address['addressLine1'],
                        'address2' => @$address['addressLine2'],
                        'address3' => @$address['addressLine3'],
                        'city' => $address['city'],
                        'province_code' => $address['stateOrProvinceCode'],
                        'zip' => $address['postalCode'],
                        'country' => $address['countryCode'],
                    ],
                ];
            }, $this->whenLoaded('inboundShipmentRelation', function () {
                return [
                    'id' => $this->inboundShipmentRelation->id,
                    'ShipmentId' => $this->inboundShipmentRelation->ShipmentId,
                    'DestinationFulfillmentCenterId' => $this->inboundShipmentRelation->DestinationFulfillmentCenterId,
                ];
            })),

            //-----------------------------------------------------
            // Fulfillment Info
            //-----------------------------------------------------

            'shipping_method_id' => $this->inclusive('shipping_method_id', $this->requested_shipping_method_id),
            'shipping_method' => $this->inclusive('shipping_method', $this->whenLoaded('requestedShippingMethod', function () {
                return $this->shippingMethod->full_name ?? null;
            })),
            'requested_shipping_method_id' => $this->inclusive('shipping_method_id', $this->requested_shipping_method_id),
            'requested_shipping_method_name' => $this->inclusive('shipping_method', $this->whenLoaded('requestedShippingMethod', function () {
                return $this->shippingMethod->full_name ?? null;
            })),
            'requested_shipping_method' => $this->inclusive('requested_shipping_method', $this->requested_shipping_method),
            'fully_received_at' => $this->inclusive('fully_received_at', $this->fully_received_at),
            'estimated_delivery_date' => $this->inclusive('estimated_delivery_date', $this->estimated_delivery_date ?? null),
            'fulfillment_info' => $this->inclusive('fulfillment_info', PurchaseOrderShipmentInfoResource::collection($this->whenLoaded('purchaseOrderShipments'))),

            //-----------------------------------------------------
            // Supplier Info
            //-----------------------------------------------------

            'supplier_id' => $this->inclusive('supplier_id', $this->supplier_id),
            $this->mergeWhen($this->resource->relationLoaded('supplier'), function () {
                return [
                    'supplier_name' => $this->inclusive('supplier_name', [
                        'id' => $this->supplier_id,
                        'name' => $this->supplier->name,
                    ]),
                    'supplier_email' => $this->inclusive('supplier_email', $this->supplier->email),
                    'supplier_po_email' => $this->inclusive('supplier_po_email', $this->supplier->purchase_order_email),
                    'supplier_company' => $this->inclusive('supplier_company', $this->supplier->company_name),
                    'supplier_contact' => $this->inclusive('supplier_contact', $this->supplier->primary_contact_name),
                    'supplier_phone' => $this->inclusive('supplier_phone', $this->supplier->address->phone ?? null),
                    'supplier_address' => $this->inclusive('supplier_address', $this->supplier->address->address1 ?? null),
                    'supplier_city' => $this->inclusive('supplier_city', $this->supplier->address->city ?? null),
                    'supplier_province' => $this->inclusive('supplier_province', $this->supplier->address->province ?? null),
                    'supplier_province_code' => $this->inclusive('supplier_province_code', $this->supplier->address->province_code ?? null),
                    'supplier_zip' => $this->inclusive('supplier_zip', $this->supplier->address->zip ?? null),
                    'supplier_country' => $this->inclusive('supplier_country', $this->supplier->address->country ?? null),
                    'supplier_country_code' => $this->inclusive('supplier_country_code', $this->supplier->address->country_code ?? null),
                ];
            }),

            //-----------------------------------------------------
            // Destination Info
            //-----------------------------------------------------

            // for frontend links
            'destination_warehouse_id' => $this->inclusive('destination_warehouse_id', $this->destination_warehouse_id),
            'destination_warehouse_type' => $this->inclusive('destination_warehouse_type', $this->whenLoaded('destinationWarehouse', function () {
                return $this->destinationWarehouse->type;
            })),
            'sales_order_id' => $this->inclusive('sales_order_id', $this->sales_order_id),
            'supplier_warehouse_id' => $this->inclusive('supplier_warehouse_id', $this->supplier_warehouse_id),
            'customer_id' => $this->inclusive('customer_id', $this->salesOrder->customer_id ?? null),

            'destination_address_id' => $this->destination_address_id,
            $this->mergeWhen($this->resource->relationLoaded('destinationAddress'), function () {
                return [
                    'destination_name' => $this->inclusive('destination_name', [
                        'id' => $this->destination->id,
                        'name' => $this->destination->name,
                    ]),
                    'destination_email' => $this->inclusive('destination_email', $this->destination->email),
                    'destination_company' => $this->inclusive('destination_company', $this->destination->company),
                    'destination_phone' => $this->inclusive('destination_phone', $this->destination->phone),
                    'destination_address' => $this->inclusive('destination_address', $this->destination->address1),
                    'destination_city' => $this->inclusive('destination_city', $this->destination->city),
                    'destination_province' => $this->inclusive('destination_province', $this->destination->province),
                    'destination_province_code' => $this->inclusive('destination_province_code', $this->destination->province_code),
                    'destination_zip' => $this->inclusive('destination_zip', $this->destination->zip),
                    'destination_country' => $this->inclusive('destination_country', $this->destination->country),
                    'destination_country_code' => $this->inclusive('destination_country_code', $this->destination->country_code),
                ];
            }),

            //-----------------------------------------------------
            // Items Info
            //-----------------------------------------------------

            'items' => $this->inclusive('items', $this->whenLoaded('purchaseOrderLines', function () {
                // set PurchaseOrder relation to prevent load it again from the DB
                $this->productLines->map(function (PurchaseOrderLine $purchaseOrderLine) {
                    $purchaseOrderLine->setRelation('purchaseOrder', $this->resource);
                });

                return PurchaseOrderLineResource::collection($this->productLines()->with(['product'])->take(3)->get())->additional(['loadedRelations' => $this->getRelations()]);
            })),

            //-----------------------------------------------------
            // Items Summary Info
            //-----------------------------------------------------

            'summary_info' => $this->whenLoaded('purchaseOrderLines', function () {
                return [
                    'total_lines' => $this->purchaseOrderLines->count(),
                    'total_product_lines' => $this->productLines()->count(),
                    'total_units' => $this->productLines()->sum('quantity'),
                    'total_quantity_pending_receipt' => $this->productLines()->sum('unreceived_quantity'),
                    'tax_summary' => $this->getTaxSummary(),
                ];
            }),

            //-----------------------------------------------------
            // Allocations Info
            //-----------------------------------------------------

            //      'allocations' => $this->inclusive( 'allocations', $this->allocations ), // ?

            //-----------------------------------------------------
            // Statuses Info
            //-----------------------------------------------------

            'order_status' => $this->inclusive('order_status', $this->order_status),
            'submission_status' => $this->inclusive('submission_status', $this->submission_status),
            'receipt_status' => $this->inclusive('receipt_status', $this->receipt_status),
            'shipment_status' => $this->inclusive('shipment_status', $this->shipment_status),
            'invoice_status' => $this->inclusive('invoice_status', $this->invoice_status),

            //-----------------------------------------------------
            // Totals Info
            //-----------------------------------------------------

            'additional_cost' => $this->inclusive('additional_cost', $this->additional_cost),
            'discount'        => $this->inclusive('discount', $this->discount),
            'product_total'   => $this->inclusive('product_total', $this->product_total),
            'total'           => $this->inclusive('total', $this->total),
            'subtotal' => $this->inclusive('total', $this->total - $this->tax_cost),
            'currency_code'   => $this->inclusive('currency_code', $this->currency->code ?? Currency::default()->code),
            'total_quantity'  => $this->inclusive('total_quantity', $this->total_quantity),

            //-----------------------------------------------------
            // Other Info
            //-----------------------------------------------------

            'tags' => $this->inclusive('tags', $this->whenLoaded('tags', fn () => $this->tags->pluck('name'))),

            //-----------------------------------------------------
            // Supplier Notes
            //-----------------------------------------------------

            'supplier_notes' => $this->inclusive('supplier_notes', $this->supplier_notes),

            'store_id' => $this->inclusive('store_id', $this->store_id),
            'store' => $this->inclusive('store', $this->whenLoaded('store', function () {
                return StoreResource::make($this->store);
            })),

            //-----------------------------------------------------
            // Linked Adjustments
            //-----------------------------------------------------
            'adjustments' => $this->inclusive('adjustments', $this->whenLoaded('adjustments', fn () => InventoryAdjustmentResource::collection($this->adjustments->toBase()->merge($this->purchaseOrderLines->pluck('adjustments')->flatten())))),

            //-----------------------------------------------------
            // Invoices
            //-----------------------------------------------------
            'invoices' => $this->inclusive('invoices', $this->whenLoaded('purchaseInvoices', function () {
                return $this->purchaseInvoices->map(function ($purchaseInvoice) {
                    return [
                        'id' => $purchaseInvoice->id,
                        'supplier_invoice_number' => $purchaseInvoice->supplier_invoice_number,
                    ];
                })->toArray();
            })),
            'tax_cost' => $this->inclusive('tax_cost', $this->tax_cost),
            'is_tax_included' => $this->is_tax_included,
            'calculated_tax_total' => $this->inclusive('calculated_tax_total', $this->calculated_tax_total),
            'tax_rate_id' => $this->inclusive('tax_rate_id', $this->tax_rate_id),
            'tax_rate' => $this->inclusive('tax_rate', $this->tax_rate),
            'asn_last_sent_at' => $this->inclusive('asn_last_sent_at', $this->asn_last_sent_at),
            'latest_estimated_delivery_date' => $this->inclusive('latest_estimated_delivery_date', $this->whenLoaded('purchaseOrderLines', function () {
                $date = $this->purchaseOrderLines->max('estimated_delivery_date');

                return $date ?? $this->estimated_delivery_date;
            }))
        ];
    }
}
