<?php

namespace App\Http\Resources;

use App\DataTable\DataTableResource;
use App\Helpers;
use App\Models\Attribute;
use App\Models\Currency;
use App\Models\SupplierPricingTier;
use App\Queries\Product;
use App\Repositories\ProductRepository;
use Illuminate\Http\Request;

/**
 * Class ProductResource.
 *
 *
 * @mixin Product
 */
class ProductResource extends DataTableResource
{
    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        /**
         * Defaults.
         */
        $defaultSupplierPricingTier = SupplierPricingTier::default();
        $defaultCurrency = Currency::default();

        return [
            //-----------------------------------------------------
            // Basic Info
            //-----------------------------------------------------

            'sku' => $this->inclusive('sku', $this->sku),
            'id' => $this->id,
            'unit_cost' => $this->inclusive('unit_cost', $this->unit_cost),
            'image' => $this->inclusive('image', fn () => $this->image),
            'image_url' => $this->inclusive('image_url', fn () => $this->image),
            'image_resolvable' => $this->inclusive('image_resolvable', fn () => $this->primaryImage->resolvable ?? null),
            'other_images' => $this->inclusive('other_images', fn () => $this->otherImages()->get(['id', 'url'])),
            'name' => $this->inclusive('name', $this->name),
            'brand' => $this->inclusive('brand', $this->whenLoaded('brand', function () {
                return [
                    'id' => $this->brand->id,
                    'name' => $this->brand->name,
                ];
            })),
            'has_inventory_movements' => $this->hasInventoryMovements(),
            'barcode' => $this->inclusive('barcode', $this->barcode),
            'mpn' => $this->inclusive('mpn', $this->mpn),
            'tags' => $this->inclusive('tags', $this->whenLoaded('tags', fn () => $this->tags->pluck('name'))),
            'type' => $this->inclusive('type', $this->type),
            'has_fba' => $this->inclusive('has_fba', fn () => $this->has_fba),
            'is_variation' => $this->inclusive('is_variation', $this->is_variation),
            'matrix_product' => $this->inclusive('matrix_product', $this->whenLoaded('parent', function () {
                return [
                    'id' => $this->parent->id,
                    'sku' => $this->parent->sku,
                ];
            })),
            'variations' => $this->inclusive('variations', $this->whenLoaded('variations', function () use ($request) {
                return $this->variations->when($request->input('supplier_id'), function ($collection) use ($request) {
                    return $collection->filter(function ($product) use ($request) {
                        return $product->suppliers->contains(function ($supplier) use ($request) {
                            return $supplier->id == $request->input('supplier_id');
                        });
                    });
                })->map(function (\App\Models\Product $product) {
                    return [
                        'id' => $product->id,
                        'sku' => $product->sku,
                        'name' => $product->name,
                        'parent_id' => $this->parent_id,
                        'price' => ProductPricingResource::collection($product->productPricingTiers->keyBy('name')),
                        'attributes' => ProductAttributeResource::collection($product->productAttributeValues),
                    ];
                });
            })),
            'shared_children_attributes' => $this->inclusive('shared_children_attributes', function () {
                return Attribute::with([])->whereIn('id', $this->shared_children_attributes ?? [])->get([
                    'id',
                    'name',
                ]);
            }),
            'proforma_landed_cost_percentage' => $this->inclusive('proforma_landed_cost_percentage', $this->proforma_landed_cost_percentage),

            //-----------------------------------------------------
            // Pricing & Cost
            //-----------------------------------------------------
            'price' => $this->inclusive('price', ProductPricingResource::collection($this->whenLoaded('productPricingTiers', function () {
                return $this->productPricingTiers->keyBy('name');
            }))),
            'default_pricing_tier_name' => $this->inclusive('default_pricing_tier_name', fn () => $this->default_pricing_tier_name),
            'average_cost' => $this->inclusive('average_cost', [
                'value' => $this->average_cost,
                'currency' => $defaultCurrency ? $defaultCurrency->code : null,
            ]),
            //-----------------------------------------------------
            // Sourcing
            //-----------------------------------------------------
            'default_supplier' => $this->inclusive('default_supplier', fn () => $this->when($this->whenLoaded('defaultSupplierProduct') && $this->defaultSupplierProduct, function () {
                return [
                    'id' => $this->defaultSupplierProduct->supplier->id ?? null,
                    'name' => $this->defaultSupplierProduct->supplier->name ?? null,
                ];
            })),
            'suppliers' => $this->inclusive('suppliers', function () {
                return $this->whenLoaded('supplierProducts', $this->getFormattedSuppliers());
            }),
            'default_supplier_sku' => $this->inclusive('default_supplier_sku', fn () => $this->whenLoaded('defaultSupplierProduct', $this->defaultSupplierProduct->supplier_sku ?? null)),
            'default_supplier_leadtime' => $this->inclusive('default_supplier_leadtime', fn () => $this->whenLoaded('defaultSupplierProduct', $this->defaultSupplierProduct->product_leadtime ?? null)),
            'default_supplier_moq' => $this->inclusive('default_supplier_moq', fn () => $this->whenLoaded('defaultSupplierProduct', $this->defaultSupplierProduct->minimum_order_quantity ?? null)),
            'default_supplier_price' => $this->inclusive('default_supplier_price', fn () => $this->whenLoaded('defaultSupplierProduct', function () use ($defaultSupplierPricingTier) {
                return $this->defaultSupplierProduct ?
                    ($this->defaultSupplierProduct->getDefaultSupplierPricing($this->defaultSupplierProduct->supplier->default_pricing_tier_id ?? $defaultSupplierPricingTier?->id)->price ?? null) :
                    null;
            })),
            'default_supplier_price_currency' => $this->inclusive('default_supplier_price_currency', fn () => $defaultSupplierPricingTier->currency_code ?? null),
            'supplier_pricing' => $this->inclusive('supplier_pricing', fn () => $this->whenLoaded('defaultSupplierProduct', SupplierProductPricingResource::collection($this->when($this->defaultSupplierProduct && $this->defaultSupplierProduct->relationLoaded('supplierPricingTiers'), function () {
                return $this->defaultSupplierProduct->supplierPricingTiers->keyBy('name');
            })))),

            //
            //      $this->mergeWhen($this->whenLoaded('defaultSupplierProduct') && $this->defaultSupplierProduct, function () use ($defaultSupplierPricingTier) {
            //        return [
            //          'default_supplier'                => $this->inclusive('default_supplier', [
            //            'id'   => $this->defaultSupplierProduct->supplier->id ?? null,
            //            'name' => $this->defaultSupplierProduct->supplier->name ?? null,
            //          ]),
            //          'default_supplier_sku'            => $this->inclusive('default_supplier_sku', $this->defaultSupplierProduct->supplier_sku),
            //          'default_supplier_leadtime'       => $this->inclusive('default_supplier_leadtime', $this->defaultSupplierProduct->product_leadtime),
            //          'default_supplier_moq'            => $this->inclusive('default_supplier_moq', $this->defaultSupplierProduct->minimum_order_quantity),
            //          'default_supplier_price'          => $this->inclusive('default_supplier_price', $this->defaultSupplierProduct->getDefaultSupplierPricing($this->defaultSupplierProduct->supplier->default_pricing_tier_id ?? ($defaultSupplierPricingTier ? $defaultSupplierPricingTier->id : null))->price ?? null),
            //          'default_supplier_price_currency' => $this->inclusive('default_supplier_price_currency', $defaultSupplierPricingTier->currency_code ?? null),
            //          'supplier_pricing'                => $this->inclusive('supplier_pricing', SupplierProductPricingResource::collection($this->when($this->defaultSupplierProduct->relationLoaded('supplierPricingTiers'), function () {
            //            return $this->defaultSupplierProduct->supplierPricingTiers->keyBy('name');
            //          }))),
            //        ];
            //      }),

            //-----------------------------------------------------
            // Components
            //-----------------------------------------------------

            'components' => $this->inclusive('components', fn () => $this->when(($this->type == Product::TYPE_BUNDLE || $this->type == Product::TYPE_KIT) && $this->whenLoaded('components'), function () {
                $components = $this->components()->get()->map(function ($component) {
                    $component->quantity = $component->pivot->quantity;

                    //unset($component->pivot);
                    return $component;
                });

                return $components;
            })),

            //-----------------------------------------------------
            // Blemished fields
            //-----------------------------------------------------
            'blemished' => $this->inclusive('blemished', fn () => $this->when($this->isBlemished() && $this->productBlemished, function () {
                return [
                    'derived_product' => [
                        'id' => $this->productBlemished->originalProduct->id,
                        'name' => $this->productBlemished->originalProduct->name,
                        'sku' => $this->productBlemished->originalProduct->sku,
                    ],
                    'condition' => $this->productBlemished->condition,
                    'reference' => $this->productBlemished->reference,
                ];
            })),

            //-----------------------------------------------------
            // Shipping Info
            //-----------------------------------------------------

            // Weight

            'weight' => $this->inclusive('weight', Helpers::getDecimalAttribute($this->weight)),
            'weight_unit' => $this->inclusive('weight_unit', $this->weight_unit),
            'weight_info' => $this->inclusive('weight_info', [
                'weight' => Helpers::getDecimalAttribute($this->weight),
                'weight_unit' => $this->weight_unit,
            ]),

            // Dimensions

            'length' => $this->inclusive('length', Helpers::getDecimalAttribute($this->length)),
            'width' => $this->inclusive('width', Helpers::getDecimalAttribute($this->width)),
            'height' => $this->inclusive('height', Helpers::getDecimalAttribute($this->height)),
            'dimension_unit' => $this->inclusive('dimensions_unit', $this->dimension_unit),
            'dimensions_info' => $this->inclusive('dimensions_info', [
                'length' => Helpers::getDecimalAttribute($this->length),
                'width' => Helpers::getDecimalAttribute($this->width),
                'height' => Helpers::getDecimalAttribute($this->height),
                'dimensions_unit' => $this->dimension_unit,
            ]),

            // Case

            'case_weight' => $this->inclusive('case_weight', Helpers::getDecimalAttribute($this->case_weight)),
            'case_weight_unit' => $this->inclusive('case_weight_unit', $this->case_weight_unit),
            'case_weight_info' => $this->inclusive('case_weight_info', [
                'weight' => Helpers::getDecimalAttribute($this->case_weight),
                'weight_unit' => $this->case_weight_unit,
            ]),

            'fba_prep_instructions' => $this->inclusive('fba_prep_instructions', $this->fba_prep_instructions),
            'case_quantity' => $this->inclusive('case_quantity', $this->case_quantity),
            'case_length' => $this->inclusive('case_length', Helpers::getDecimalAttribute($this->case_length)),
            'case_width' => $this->inclusive('case_width', Helpers::getDecimalAttribute($this->case_width)),
            'case_height' => $this->inclusive('case_height', Helpers::getDecimalAttribute($this->case_height)),
            'case_dimension_unit' => $this->inclusive('case_dimensions_unit', $this->case_dimension_unit),
            'case_dimensions' => $this->inclusive('case_dimensions', [
                'length' => Helpers::getDecimalAttribute($this->case_length),
                'width' => Helpers::getDecimalAttribute($this->case_width),
                'height' => Helpers::getDecimalAttribute($this->case_height),
                'dimensions_unit' => $this->case_dimension_unit,
            ]),

            //-----------------------------------------------------
            // Taxonomy
            //-----------------------------------------------------
            'category_main' => $this->inclusive('category_main', $this->whenLoaded('primaryCategory', function () {
                return $this->primaryCategory->first() ? $this->primaryCategory->first()->only('id', 'name') : null;
            })),
            'category_main_path' => $this->inclusive('category_main_path', $this->whenLoaded('primaryCategory', function () {
                return collect($this->getCategoryPath())->keyBy('id');
            })),
            //
            //      $this->mergeWhen($this->whenLoaded('primaryCategory', true, false), function () {
            //        return [
            //          'category_main'      => $this->inclusive('category_main', function () {
            //            return $this->primaryCategory->first() ? $this->primaryCategory->first()->only('id', 'name') : null;
            //          }),
            //          'category_main_path' => $this->inclusive('category_main_path', collect($this->getCategoryPath())->keyBy('id')),
            //        ];
            //      }),

            'category_others' => $this->inclusive('category_others', $this->whenLoaded('otherCategories', function () {
                return ProductCategoryResource::collection($otherCategories = $this->otherCategories);
            })),
            'category_others_path' => $this->inclusive('category_others_path', $this->whenLoaded('otherCategories', function () {
                $paths = [];
                if (empty($otherCategories)) {
                    $otherCategories = $this->otherCategories;
                }
                foreach ($otherCategories as $category) {
                    $paths[] = $this->getCategoryPath($category);
                }

                return collect($paths)->toArray();
            })),

            //      $this->mergeWhen($this->whenLoaded('otherCategories', true, false), function () {
            //        return [
            //          'category_others'      => $this->inclusive('category_others', ProductCategoryResource::collection($otherCategories = $this->otherCategories)),
            //          'category_others_path' => $this->inclusive('category_others_path', function () use ($otherCategories) {
            //            $paths = [];
            //            if (empty($otherCategories)) {
            //              $otherCategories = $this->otherCategories;
            //            }
            //            foreach ($otherCategories as $category) {
            //              $paths[] = $this->getCategoryPath($category);
            //            }
            //
            //            return collect($paths)->toArray();
            //          }),
            //        ];
            //      }),

            //-----------------------------------------------------
            // Inventory
            //-----------------------------------------------------

            'inventory_total' => $this->inclusive('inventory_total', $this->whenLoaded('totalInventory', function () {
                return $this->totalInventory->inventory_total;
            }, $this->inventory_total)),
            'inventory_available' => $this->inclusive('inventory_available', $this->whenLoaded('totalInventory', function () {
                return $this->totalInventory->inventory_available;
            }, $this->inventory_available)),
            'inventory_reserved' => $this->inclusive('inventory_reserved', $this->whenLoaded('totalInventory', function () {
                return $this->totalInventory->inventory_reserved;
            }, $this->inventory_reserved)),
            'inventory_incoming' => $this->inclusive('inventory_incoming', $this->whenLoaded('purchaseOrderLines', function () {
                return app(ProductRepository::class)->getInboundQuantityForProduct($this->getModel());
            }, 0)),
            'inventory_in_transit' => $this->inclusive('inventory_in_transit', $this->whenLoaded('totalInventory', function () {
                return $this->totalInventory->inventory_in_transit;
            }, $this->inventory_in_transit)),
            'inventory_stock_value' => $this->inclusive('inventory_stock_value', $this->whenLoaded('totalInventory', function () {
                return $this->totalInventory->inventory_stock_value;
            }, $this->inventory_stock_value)),
            'daily_average_consumption' => $this->inclusive('daily_average_consumption', $this->daily_average_consumption),
            // inventory in warehouses
            'inventory' => $this->inclusive('inventory', $this->whenLoaded('warehousesInventory', function () {
                return ProductInventoryResource::collection($this->warehousesInventory->keyBy('warehouse.name'));
            })),
            'inventory_available_warehouses' => $this->inclusive('inventory_available_warehouses', $this->whenLoaded('warehousesInventory', function () {
                return ProductInventoryAvailableResource::collection($this->warehousesInventory->keyBy('warehouse.name'));
            })),

            //-----------------------------------------------------
            // Attributes
            //-----------------------------------------------------
            'attributes' => $this->inclusive('attributes', fn () => ProductAttributeResource::collection($this->productAttributeValues->keyBy('attribute.name'))),

            //-----------------------------------------------------
            // Timestamps
            //-----------------------------------------------------

            'archived_at' => $this->inclusive('archived_at', $this->archived_at),
            'created_at' => $this->inclusive('created_at', $this->created_at),
            'updated_at' => $this->inclusive('updated_at', $this->updated_at),

            //-----------------------------------------------------
            // Nominal Codes
            //-----------------------------------------------------

            'sales_nominal_code' => $this->inclusive('sales_nominal_code', $this->whenLoaded('salesNominalCode', fn () => $this->salesNominalCode?->name)),
            'sales_nominal_code_id' => $this->inclusive('sales_nominal_code', $this->sales_nominal_code_id),
            'cogs_nominal_code' => $this->inclusive('cogs_nominal_code', $this->whenLoaded('cogsNominalCode', fn () => $this->cogsNominalCode?->name)),
            'cogs_nominal_code_id' => $this->inclusive('cogs_nominal_code_id', $this->cogs_nominal_code_id),
            'is_taxable' => $this->inclusive('is_taxable', $this->is_taxable),
            'default_tax_rate_id' => $this->inclusive('default_tax_rate_id', $this->default_tax_rate_id),
            'component_skus' => $this->inclusive('component_skus', $this->whenLoaded('components', function () {
                return $this->components->map(function ($component) {
                    return $component->sku.'->'.$component->pivot->quantity;
                })->implode(', ');
            })),
            'component_ids' => $this->inclusive('component_ids', $this->whenLoaded('components', function () {
                return $this->components->map(function ($component) {
                    return $component->id.'->'.$component->pivot->quantity;
                })->implode(', ');
            })),
            'component_of_sku' => $this->inclusive('component_of_sku', $this->whenLoaded('parentProducts', function () {
                return $this->parentProducts->map(function ($parentProduct) {
                    return $parentProduct->sku.'->'.$parentProduct->pivot->quantity;
                })->implode(',');
            })),
            'component_of_id' => $this->inclusive('component_of_id', $this->whenLoaded('parentProducts', function () {
                return $this->parentProducts->map(function ($parentProduct) {
                    return $parentProduct->id.'->'.$parentProduct->pivot->quantity;
                })->implode(',');
            })),
            'notes' => $this->inclusive('notes', $this->whenLoaded('notes', function () {
                return $this->notes->map(function ($note) {
                    return [
                        'id' => $note->id,
                        'note' => $note->note,
                        'created_at' => $note->created_at,
                        'user' => $note->user,
                    ];
                });
            })),

        ];
    }

    public function getFormattedSuppliers(): string
    {
        $formattedSuppliers = [];
        $this->supplierProducts->each(function ($product) use (&$formattedSuppliers) {
            $supplier = [
                'name' => $product->supplier->name,
                'supplier_sku' => $product->supplier_sku,
                'supplier_pricing' => $product->supplierPricingTiers->filter(function ($tier) {
                    return $tier->pivot->price > 0;
                })->map(function ($tier) {
                    return [
                        'name' => $tier->name,
                        'price' => $tier->pivot->price,
                        'currency_code' => $tier->currency_code,
                    ];
                })->values()->toArray(),
                'minimum_order_quantity' => $product->minimum_order_quantity,
                'leadtime' => $product->leadtime,
                'is_default' => $product->is_default ?? null,
            ];
            // Filter out null and false values from the supplier array
            $supplier = array_filter($supplier, function ($value) {
                return $value !== null && $value !== false;
            });

            $formattedSuppliers[] = $supplier;
        });

        return json_encode($formattedSuppliers);
    }
}
