<?php

namespace App\Http\Resources;

use App\Models\Product;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Repositories\ProductRepository;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Repositories\AmazonProductRepository;

/**
 * Class ProductInventoryDetailsResource.
 *
 *
 * @mixin Product
 */
class ProductInventoryDetailsResource extends JsonResource
{
    private AmazonProductRepository $amazonProductRepository;

    public function __construct($resource)
    {
        parent::__construct($resource);
        $this->amazonProductRepository = app(AmazonProductRepository::class);
    }

    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        $no_fba = filter_var($request->input('no_fba', false), FILTER_VALIDATE_BOOLEAN);
        $noSuppliers = filter_var($request->input('no_suppliers', false), FILTER_VALIDATE_BOOLEAN);
        // 1- if the supplier is sourcing and dropship is enabled
        // 2- if not supplier warehouses we will add all of them (not necessarily have movements)

        $productWarehouses = Warehouse::with(['integrationInstance.salesChannel'])
            ->where(function ($query) use ($noSuppliers) {
                $query->whereNull('supplier_id');
                if (! $noSuppliers) {
                    $query->orWhereIn('id', $this->getSupplierDropshipWarehouses());
                }
            })
            ->when($no_fba, function ($query) {
                return $query->where('type', '!=', Warehouse::TYPE_AMAZON_FBA);
            })
            ->archived(0)
            ->get(['id', 'name', 'supplier_id', 'type', 'integration_instance_id']);

        $productWarehouses = $productWarehouses->map(function ($warehouse) {
            if ($warehouse->type === Warehouse::TYPE_AMAZON_FBA) {
                $warehouse->name .= ' (real-time)';
            }

            return $warehouse;
        });

        if ($no_fba) {
            $warehousesFBARealTimeData = collect([]);
        } else {
            $integrationInstanceIds = $productWarehouses->where('type', Warehouse::TYPE_AMAZON_FBA)->pluck('integration_instance_id');
            $warehousesFBARealTimeData = $this->amazonProductRepository->getFBAReportInventoryForIntegrationInstances($this->id, $integrationInstanceIds);
        }

        //    $incomingQuantity = $this->getIncomingQuantity();
        $inventory = [
            'total' => [
                'inbound' => $this->purchaseOrderLinesOpened->sum('unreceived_inventory') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'inbound'),
                'stock' => $this->inWarehousesQuantity->sum('inventory_total') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData),
                'reserved' => $this->inWarehousesReservedQuantity->sum('inventory_reserved') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'reserved'),
                'available' => $this->inWarehouseAvailableQuantity->sum('inventory_available') + $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'available'),
                'in_transit' => $this->inWarehouseTransitQuantity->sum('inventory_in_transit'),
            ],
            'unit_cost' => $this->unit_cost,
        ];

        $inventory['warehouses'] = [];

        /** @var Warehouse $warehouse */
        foreach ($productWarehouses as $warehouse) {
            $supplierInventory = $this->suppliersInventory->firstWhere('warehouse_id', $warehouse->id);
            if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA && $warehouse->integrationInstance) {
                $listingCount = $this->productListings->where('is_fba', $warehouse->type == Warehouse::TYPE_AMAZON_FBA)->where('sales_channel_id', $warehouse->integrationInstance->salesChannel->id)->count();
            } else {
                $listingCount = 0;
            }
            if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                $inventory['warehouses'][$warehouse->name] = [
                    'inbound' => $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'inbound', $warehouse->integrationInstance),
                    'stock' => $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'total', $warehouse->integrationInstance),
                    'reserved' => $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'reserved', $warehouse->integrationInstance),
                    'available' => $this->amazonProductRepository->extractRealTimeInventory($warehousesFBARealTimeData, 'available', $warehouse->integrationInstance),
                    'in_transit' => 0,
                    'warehouse_id' => $warehouse->id,
                    'warehouse_name' => $warehouse->name,
                    'warehouse_type' => $warehouse->type,
                    'initial_inventory' => [],
                    'supplier' => null,
                    'listing_count' => $listingCount,
                ];
            } else {
                $inventory['warehouses'][$warehouse->name] = [
                    'inbound' => app(ProductRepository::class)->getInboundQuantityForProduct($this->getModel(), $warehouse),
                    'stock' => $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) :
                        $this->inWarehousesQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_total'),
                    'reserved' => $this->inWarehousesReservedQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_reserved'),
                    'available' => $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) :
                        $this->inWarehouseAvailableQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_available'),
                    'in_transit' => $this->inWarehouseTransitQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_in_transit'),
                    'warehouse_id' => $warehouse->id,
                    'warehouse_name' => $warehouse->name,
                    'initial_inventory' => $warehouse->initialInventoryForProduct($this->id, $this->unit_cost),
                    'supplier' => $warehouse->isSupplierWarehouse() ? [
                        'name' => $warehouse->supplier->name,
                        'id' => $warehouse->supplier->id,
                    ] : null,
                    'listing_count' => $listingCount,
                    'warehouse_type' => $warehouse->type,
                ];
            }
        }
        $inventory = array_merge($inventory, $this->potentialBundles($this->id));

        return $inventory;
    }

    private function getSupplierDropshipWarehouses()
    {
        $suppliersDropshipWarehousesId = [];
        $this->suppliers->map(function (Supplier $supplier) use (&$suppliersDropshipWarehousesId) {
            $dropshipWarehouses = $supplier->warehouses->where('dropship_enabled', true);
            if ($dropshipWarehouses->isNotEmpty()) {
                $suppliersDropshipWarehousesId = array_merge($suppliersDropshipWarehousesId, $dropshipWarehouses->pluck('id')->toArray());
            }
        });

        return $suppliersDropshipWarehousesId;
    }

    public function potentialBundles($productId)
    {
        $product = Product::find($productId);

        if ($product->type !== 'bundle') {
            return ['bundle_data' => []];
        }

        $potentialBundles = $product->components()
            ->with('productInventory')
            ->get()
            ->flatMap(function ($component) {
                return $component->productInventory->map(function ($warehouse) use ($component) {
                    return [
                        'warehouse_id' => $warehouse['warehouse_id'],
                        'inventory_reserved' => [
                            'max_bundle_quantity' => intval($warehouse['inventory_reserved'] / $component->pivot->quantity),
                        ],
                        'inventory_incoming' => [
                            'max_bundle_quantity' => intval($warehouse['inventory_incoming'] / $component->pivot->quantity),
                        ],
                        'inventory_in_transit' => [
                            'max_bundle_quantity' => intval($warehouse['inventory_in_transit'] / $component->pivot->quantity),
                        ],
                        'inventory_available' => [
                            'max_bundle_quantity' => intval($warehouse['inventory_available'] / $component->pivot->quantity),
                        ],
                    ];
                });
            })
            ->groupBy('warehouse_id')
            ->map(function ($group) {
                return [
                    'warehouse_id' => $group[0]['warehouse_id'],
                    'inventory_reserved' => ['max_bundle_quantity' => $group->min('inventory_reserved.max_bundle_quantity')],
                    'inventory_incoming' => ['max_bundle_quantity' => $group->min('inventory_incoming.max_bundle_quantity')],
                    'inventory_in_transit' => ['max_bundle_quantity' => $group->min('inventory_in_transit.max_bundle_quantity')],
                    'inventory_available' => ['max_bundle_quantity' => $group->min('inventory_available.max_bundle_quantity')],
                ];
            });

        return ['bundle_data' => $potentialBundles];
    }
}
