<?php

namespace App\Http\Resources;

use App\Models\InventoryForecastItem;
use App\Models\PurchaseOrderLine;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

/**
 * Class InventoryForecastResource.
 *
 * @mixin InventoryForecastItem
 */
class InventoryForecastItemResource extends JsonResource
{

    /**
     * Transform the resource into an array.
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'out_of_stock_date' => $this->out_of_stock_date,
            'forecast_date' => $this->forecast_date,
            'suggested_purchase_quantity' => $this->suggested_purchase_quantity,
            'total_cost' => $this->unit_cost * $this->suggested_purchase_quantity,
            'unit_cost' => (float)$this->unit_cost,
            'unit_cost_in_tenant_currency' => (float)$this->unit_cost_in_tenant_currency,
            'tenant_currency' => [
                'id' => $this->tenant_currency_id,
                'code' => $this->tenant_currency_code,
            ],
            'available_stock' => $this->available_stock,
            'inbound_stock' => $this->inbound_stock,
            'in_transit_stock' => $this->in_transit_stock,
            'current_inventory' => [
                'total' => [
                    'inbound' => $this->inventory_inbound,
                    'available' => $this->inventory_available,
                    'in_transit' => $this->inventory_in_transit,
                ]
            ],
            'supplier' => $this->whenLoaded('supplier', function () {
                return [
                    'id' => $this->supplier_id,
                    'name' => $this->supplier->name,
                    'leadtime' => $this->supplier->leadtime,
                    'default_pricing_tier' =>  $this->supplier->defaultPricingTier ? [
                        'id' => $this->supplier->defaultPricingTier->id,
                        'name' => $this->supplier->defaultPricingTier->name,
                        'currency_code' => $this->supplier->defaultPricingTier->currency_code,
                    ] : []
                ];
            }),
            'product' => $this->whenLoaded('product', function () {
                return [
                    'id' => $this->product_id,
                    'sku' => $this->product->sku,
                    'name' => $this->product->name,
                    'image_url' => $this->product->primaryImage->url ?? null,
                ];
            }),
            'total_quantity_purchased' => $this->whenLoaded('purchaseOrderLines', function () {
                return $this->purchaseOrderLines->sum('quantity');
            }),
            'purchase_order_lines' => $this->whenLoaded('purchaseOrderLines', function () {
                return $this->purchaseOrderLines->map(function (PurchaseOrderLine $purchaseOrderLine) {
                    return [
                        'id' => $purchaseOrderLine->id,
                        'quantity' => $purchaseOrderLine->quantity,
                    ];
                })->toArray();
            }),
            'warehouse' => $this->whenLoaded('warehouse', function () {
                return WarehouseResource::make($this->warehouse);
            }),
        ];
    }
}
