<?php

namespace App\Http\Requests;

use App\Models\WarehouseTransfer;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

/**
 * Class WarehouseTransferShipmentRequest.
 */
class WarehouseTransferShipmentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'cost' => 'nullable|numeric|min:0|lt:100000',
            'tracking_number' => 'nullable',
            'products' => 'required|array',
            'products.*.id' => 'required|exists:products,id|exists:warehouse_transfer_lines,product_id',
            'products.*.quantity' => 'required|min:0',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'shipment_date' => 'required|date',
        ];
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            // Make sure total shipment isn't more than quantity for
            // each product.
            $validator->after(function (Validator $validator) {
                $products = $validator->attributes()['products'];

                /** @var WarehouseTransfer $warehouseTransfer */
                $warehouseTransfer = WarehouseTransfer::with([])->findOrFail($this->route('transfer'));

                if ($warehouseTransfer->isDraft()) {
                    $validator->addFailure('transfer', 'MustBeOpen');

                    return;
                }

                foreach ($products as $key => $product) {
                    if ($this->shipmentExceedsTransferQuantity($product['id'], $product['quantity'], $warehouseTransfer)) {
                        $validator->addFailure('products.'.$key, 'MustNotExceedTotalQuantity');
                    }
                }
            });
        }
    }

    private function shipmentExceedsTransferQuantity(
        $productId,
        $thisShipmentQuantity,
        WarehouseTransfer $warehouseTransfer
    ): bool
    {
        // Get the total shipped for the product and compare with the total quantity of the warehouse transfer
        return $warehouseTransfer->totalShippedForProduct($productId) + $thisShipmentQuantity >
            $warehouseTransfer->totalQuantityForProduct($productId);
    }

    public function messages(): array
    {
        return [
            'must_not_exceed_total_quantity' => 'Shipment quantity for product must not exceed total quantity for warehouse transfer.',
            'must_be_open' => 'Warehouse transfer must be open to receive shipments.',
        ];
    }
}
