<?php

namespace App\Http\Requests;

use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Validator;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\FormRequest;

/**
 * Class WarehouseTransferShipmentReceiptRequest.
 */
class WarehouseTransferShipmentReceiptRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'receipt_date' => 'required|date|before_or_equal:'.Carbon::now(),
            'products' => 'required|array',
            'products.*.id' => 'required|exists:products,id|exists:warehouse_transfer_lines,product_id',
            'products.*.quantity' => 'required|numeric|gt:0',
            'blemished' => 'nullable|array',
            'blemished.*.product_id' => 'required|exists:products,id',
            'blemished.*.sku' => 'required|unique:products,sku',
            'blemished.*.quantity' => 'required|numeric|gt:0',
            'blemished.*.condition' => 'nullable|string|max:255',
            'blemished.*.reference' => 'nullable|string|max:255',
        ];

        if ($this->isMethod('put')) {
            $rules['receipt_date'] = 'sometimes|'.$rules['receipt_date'];
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            // Make sure total shipment receipt isn't more than shipment quantity for
            // each product.
            $validator->after(function (Validator $validator) {
                $products = $validator->attributes()['products'];

                /** @var WarehouseTransfer $warehouseTransfer */
                $warehouseTransfer = WarehouseTransfer::with([])->findOrFail($this->route('transfer'));

                if ($warehouseTransfer->isDraft()) {
                    $validator->addFailure('transfer', 'MustBeOpen');

                    return;
                }

                foreach ($products as $key => $product) {
                    if ($this->quantityExceedsShipmentQuantity($product['id'], $product['quantity'], $warehouseTransfer)) {
                        $validator->addFailure('products.'.$key, 'MustNotExceedTotalShippedQuantity', [
                            'field' => 'Receipt'
                        ]);
                    }
                }

                $blemished = $validator->attributes()['blemished'] ?? [];

                if(!empty($blemished)){
                    foreach ($blemished as $key => $product) {
                        if ($this->quantityExceedsShipmentQuantity($product['product_id'], $product['quantity'], $warehouseTransfer)) {
                            $validator->addFailure('blemished.'.$key, 'MustNotExceedTotalShippedQuantity', [
                                'field' => 'Blemished'
                            ]);
                        }
                    }
                }
            });
        }
    }

    private function quantityExceedsShipmentQuantity($productId, $thisReceiptQuantity, WarehouseTransfer $warehouseTransfer): bool
    {
        // For updates to receipts, we need to adjust for the originally received quantity
        $updateAdjustment = 0;
        if ($this->route('receipt')) {
            $receipt = WarehouseTransferShipmentReceipt::with([])->findOrFail(e($this->route('receipt')));
            $updateAdjustment = $receipt->receiptLines()->whereHas('shipmentLine.warehouseTransferLine', function (Builder $builder) use ($productId) {
                return $builder->where('product_id', $productId);
            })->sum('quantity');
        }

        // Get the total received for the product and compare with the total quantity shipped for the product
        return $warehouseTransfer->totalQuantityReceivedForProduct($productId) + $thisReceiptQuantity - $updateAdjustment > $warehouseTransfer->totalShippedForProduct($productId);
    }

    public function messages(): array
    {
        return [
            'must_not_exceed_total_shipped_quantity' => ':field quantity for product must not exceed total quantity shipped for product.',
            'receipt_date_before_or_equal' => 'Receipt date cannot be in the future.',
            'must_be_open' => 'Warehouse transfer must be open to receive shipments.',
        ];
    }
}
