<?php

namespace App\Http\Requests;

use App\Jobs\JasperExport;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class StoreStore extends FormRequest
{
    use JasperExport;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255|unique:stores,name',
            'email' => 'required|email|max:255',
            'po_email' => 'nullable|email|max:255',
            'company_name' => 'nullable|max:255',
            'logo_url' => 'nullable',
            'download_logo' => 'nullable',
            'custom_jrxml_file' => 'nullable',
            'custom_message' => 'nullable',
            'address1' => 'nullable|max:255',
            'address2' => 'nullable|max:255',
            'address3' => 'nullable|max:255',
            'country' => 'nullable|max:255',
            'country_code' => 'nullable|max:2',
            'zip' => 'nullable|max:255',
            'city' => 'nullable|max:255',
            'province' => 'nullable|max:255',
            'province_code' => 'nullable|max:255',
            'phone' => 'nullable|max:255',
            'fax' => 'nullable|max:255',
        ];

        if ($this->isMethod('PUT')) {
            $rules['name'] = 'sometimes|required|max:255|unique:stores,name,'.$this->route()->originalParameter('store');
            $rules['email'] = 'sometimes|'.$rules['email'];
        }

        return $rules;
    }

    public function withValidator($validator)
    {
        if ($validator->passes()) {
            $validator->after(
                function (Validator $validator) {
                    $attributes = $validator->attributes();

                    if (! empty($attributes['custom_jrxml_file']) && ! $this->checkCustomJrxmlFile()) {
                        $validator->errors()->add('custom_jrxml_file', 'Error Jasper file is not valid.');
                    }
                }
            );
        }
    }

    public function checkCustomJrxmlFile()
    {
        try {
            $templateFile = $this->input('custom_jrxml_file');
            $orderPSFile = Storage::disk('order_packing_slips')->path(Str::random());
            $dataArray = [
                [
                    'fulfillable' => true,
                    'sku' => '9429590380049',
                    'product_image' => null,
                    'description' => 'Et dolor vitae commodi mollitia quos tempora.',
                    'qty' => 6.0,
                    'barcode' => '5118381570434',
                    'mpn' => '',
                    'grouping_id' => 338,
                    'order_number' => 'SO-6358',
                    'order_date' => 'Aug 25, 2023',
                    'logo' => null,
                    'store_address' => '',
                    'shipping_address' => "7046 Jules Views\n\nSuite 467\n\nAmiyaview, MI  85987\n\nBenin\n\n+1.808.469.5875",
                    'billing_address' => "7046 Jules Views\n\nSuite 467\n\nAmiyaview, MI  85987\n\nBenin\n\n+1.808.469.5875",
                    'shipping_method' => null,
                    'store_name' => 'Casper PLC',
                    'shipping_address_name' => 'Miss Janelle Mayer',
                    'billing_address_name' => 'Carson Zieme MD',
                    'notes' => '',
                    'store_message' => '',
                ],
            ];

            $fileName = $this->export($dataArray, $this->getTemplatePath($templateFile), $orderPSFile);

            return Str::endsWith($fileName, '.pdf');
        } catch (\Exception $e) {
            return false;
        }
    }

    protected function defaultJasperTemplateFile(): string
    {
        return 'SKU_Packing_Slip.jrxml';
    }
}
