<?php

namespace App\Http\Requests;

use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class StoreShippingMethod extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255',
            'shipping_carrier_id' => 'required|exists:shipping_carriers,id',
            'delivery_min' => 'nullable|integer|min:0',
            'delivery_max' => 'nullable|integer',
            'included_locations' => 'nullable|array',
            'included_locations.*' => 'exists:constants_countries,code',
            'excluded_locations' => 'nullable|array',
            'excluded_locations.*' => 'exists:constants_countries,code',
            'type' => 'required|in:'.implode(',', ShippingMethod::TYPES),
        ];

        if ($this->isMethod('PUT')) {
            $rules['name'] = 'sometimes|'.$rules['name'];
            $rules['shipping_carrier_id'] = 'sometimes|'.$rules['shipping_carrier_id'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check shipping method name is unique by shipping carrier
                if (! empty($attributes['name'])) {
                    // get shipping carrier
                    if (! empty($attributes['shipping_carrier_id'])) {
                        $shippingCarrier = ShippingCarrier::with(['shippingMethods'])->findOrFail($attributes['shipping_carrier_id']);
                    } else {
                        /** @var ShippingMethod $shippingMethod */
                        $shippingMethod = $this->route('shipping_method');
                        $shippingMethod->load(['shippingCarrier', 'shippingCarrier.shippingMethods']);

                        $shippingCarrier = $shippingMethod->shippingCarrier;
                    }

                    // shipping methods that have the same name
                    $shippingMethods = $shippingCarrier->shippingMethods->where('name', $attributes['name']);
                    if ($shippingMethods->isNotEmpty()) {
                        if ($this->isMethod('POST') || ($this->isMethod('PUT') && $shippingMethods->first()->id != $this->route()->originalParameter('shipping_method'))) {
                            $validator->addFailure('name', 'MustBeUniqueByCarrier');
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'must_be_unique_by_carrier' => __('messages.shipping_methods.name_unique_by_carrier'),
        ];
    }
}
