<?php

namespace App\Http\Requests;

use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrderLine;
use App\Validator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\FormRequest;

class StorePurchaseInvoice extends FormRequest
{
    use InstanceFromCustom;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'purchase_order_id' => 'required|exists:purchase_orders,id',
            'purchase_invoice_date' => 'required|date',
            'supplier_invoice_number' => 'required|max:255',
            'status' => 'in:'.implode(',', PurchaseInvoice::STATUS),

            // invoice lines

            'purchase_invoice_lines' => 'required|array|min:1',
            'purchase_invoice_lines.*.purchase_order_line_id' => 'nullable|exists:purchase_order_lines,id',
            'purchase_invoice_lines.*.purchase_order_line_reference' => 'nullable|exists:purchase_order_lines,line_reference',
            'purchase_invoice_lines.*.quantity_invoiced' => 'required|numeric|min:1|lt:100000',

            // create/update original purchase order lines

            'purchase_invoice_lines.*.description' => 'required_without_all:purchase_invoice_lines.*.purchase_order_line_id,purchase_invoice_lines.*.purchase_order_line_reference|max:255',
            'purchase_invoice_lines.*.product_id' => 'nullable|exists:products,id',
            'purchase_invoice_lines.*.amount' => 'required_without_all:purchase_invoice_lines.*.purchase_order_line_id,purchase_invoice_lines.*.purchase_order_line_reference|numeric|gt:0|lt:100000',
            'purchase_invoice_lines.*.tax' => 'nullable|numeric|lt:100000',
            'purchase_invoice_lines.*.discount' => 'nullable|numeric|lt:100000',
            'purchase_invoice_lines.*.nominal_code_id' => 'nullable|exists:nominal_codes,id',
        ];

        if ($this->isMethod('put')) {
            $rules['purchase_order_id'] = 'sometimes|'.$rules['purchase_order_id'];
            $rules['purchase_invoice_date'] = 'sometimes|'.$rules['purchase_invoice_date'];
            $rules['supplier_invoice_number'] = 'sometimes|'.$rules['supplier_invoice_number'];
            $rules['purchase_invoice_lines'] = 'sometimes|'.$rules['purchase_invoice_lines'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check purchase invoice lines belong to the purchase order and uninvoiced
                if (! empty($attributes['purchase_invoice_lines'])) {
                    // Aggregate invoice lines quantities based on id's.
                    // This ensures that the user does not create more invoice
                    // quantities than possible
                    $attributes['purchase_invoice_lines'] = PurchaseInvoice::aggregateInvoiceLines($attributes['purchase_invoice_lines']);

                    $query = PurchaseOrderLine::with(['purchaseInvoiceLines'])
                        ->whereIn('id', array_column($attributes['purchase_invoice_lines'], 'purchase_order_line_id'));

                    if ($this->isMethod('put')) {
                        $query = $query->orWhere(function (Builder $builder) use ($attributes) {
                            $builder->where('purchase_order_id', PurchaseInvoice::with([])->findOrFail($this->route('purchase_invoice'))->purchase_order_id);
                            $builder->whereIn('description', array_column($attributes['purchase_invoice_lines'], 'description'));
                        });
                    }

                    $purchaseOrderLines = $query->get();

                    foreach ($attributes['purchase_invoice_lines'] as $index => $invoiceLine) {
                        if (! $invoiceLine['purchase_order_line_id']) {
                            continue;
                        } // Skip invoice lines without purchase order line ids.

                        $hasLineReference = isset($shipmentLine['purchase_order_line_reference']);
                        $key = ! $hasLineReference ? 'purchase_order_line_id' : 'purchase_order_line_reference';

                        /** @var PurchaseOrderLine $purchaseOrderLine */
                        $purchaseOrderLine = $purchaseOrderLines
                            ->firstWhere('id', $invoiceLine['purchase_order_line_id']);

                        if (! $purchaseOrderLine) {
                            // Try finding by description
                            $purchaseOrderLine = $purchaseOrderLines->firstWhere('description', $invoiceLine['description']);
                        }

                        if ($this->isMethod('post') && $purchaseOrderLine->purchase_order_id != $attributes['purchase_order_id']) {
                            $validator->addFailure("purchase_invoice_lines.{$index}.{$key}", 'IsNotBelongsToPurchaseOrder');
                        }

                        if ($this->isMethod('post')) {
                            if ($purchaseOrderLine->fully_invoiced) {
                                $validator->addFailure("purchase_invoice_lines.{$index}.{$key}", 'IsInvoiced');
                            } elseif ($invoiceLine['quantity_invoiced'] > $purchaseOrderLine->uninvoiced_quantity) {
                                $validator->addFailure("purchase_invoice_lines.{$index}.quantity_invoiced", 'IsGreaterThanUninvoicedQuantity');
                            }
                        } elseif ($this->isMethod('put')) {
                            // Updating invoice
                            if ($purchaseOrderLine->quantity < $invoiceLine['quantity_invoiced']) {
                                $validator->addFailure("purchase_invoice_lines.{$index}.quantity_invoiced", 'IsGreaterThanUninvoicedQuantity');
                            }
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_belongs_to_purchase_order' => __('messages.purchase_order.is_not_belongs'),
            'is_invoiced' => __('messages.purchase_order.fully_invoiced_line'),
            'is_greater_than_uninvoiced_quantity' => __('messages.purchase_order.greater_than_uninvoiced'),
        ];
    }
}
