<?php

namespace App\Http\Requests;

use App\Helpers;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Validator;

class StoreProduct extends FormRequest
{
    use InstanceFromCustom;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $operationsForObject = [Helpers::OPERATION_UPDATE_CREATE, Helpers::OPERATION_DELETE];
        $operationsForArray = [Helpers::OPERATION_SET, Helpers::OPERATION_APPEND, Helpers::OPERATION_DELETE];

        $rules = [
            'parent_id' => 'nullable|exists:products,id|not_in:'.$this->route('product'),
            'sku' => 'required|max:255|unique:products,sku,'.$this->route('product'),
            'barcode' => 'nullable|max:255',
            'brand_name' => 'nullable|max:255',
            'type' => 'required|in:'.implode(',', Product::TYPES),
            'weight' => 'nullable|numeric|lt:1000000',
            'weight_unit' => 'nullable|in:'.implode(',', Product::WEIGHT_UNITS),
            'length' => 'nullable|numeric|lt:1000000',
            'width' => 'nullable|numeric|lt:1000000',
            'height' => 'nullable|numeric|lt:1000000',
            'dimension_unit' => 'nullable|in:'.implode(',', Product::DIMENSION_UNITS),
            'name' => 'nullable|max:255',
            'fba_prep_instructions' => 'nullable|max:255',
            'case_quantity' => 'nullable|numeric|lt:10000',
            'case_length' => 'nullable|numeric|lt:10000',
            'case_width' => 'nullable|numeric|lt:10000',
            'case_height' => 'nullable|numeric|lt:10000',
            'case_dimension_unit' => 'nullable|in:'.implode(',', Product::DIMENSION_UNITS),
            'case_weight' => 'nullable|numeric|lt:10000',
            'case_weight_unit' => 'nullable|in:'.implode(',', Product::WEIGHT_UNITS),
            'sales_nominal_code_id' => 'nullable|exists:nominal_codes,id',
            'cogs_nominal_code_id' => 'nullable|exists:nominal_codes,id',
            'default_tax_rate_id' => 'nullable|exists:tax_rates,id',
            'proforma_landed_cost_percentage' => 'nullable|numeric',
            'proforma_shipping_cost' => 'nullable|numeric',
            'images' => 'array',
            'images.*.url' => 'required',
            'images.*.name' => 'nullable|max:255',
            'images.*.sort_order' => 'nullable|integer',
            'images.*.download' => 'nullable|boolean',
            'images.*.is_primary' => 'nullable|boolean',
            'tags' => 'nullable|array',
            'tags_operation' => 'nullable|in:'.implode(',', $operationsForArray),

            /**
             * Pricing.
             */
            'pricing' => 'nullable|array',
            'pricing.*.product_pricing_tier_id' => 'required_without:pricing.*.product_pricing_tier_name|exists:product_pricing_tiers,id',
            'pricing.*.product_pricing_tier_name' => 'required_without:pricing.*.product_pricing_tier_id|max:255',
            'pricing.*.price' => 'numeric|lt:100000|required_unless:pricing.*.operation,'.Helpers::OPERATION_DELETE,
            'pricing.*.operation' => 'nullable|in:'.implode(',', $operationsForObject),

            /**
             * Source.
             */
            'suppliers' => 'nullable|array',
            'suppliers.*.supplier_id' => 'required_without:suppliers.*.supplier_name|exists:suppliers,id',
            'suppliers.*.supplier_name' => 'required_without:suppliers.*.supplier_id|exists:suppliers,name',
            'suppliers.*.is_default' => 'nullable|boolean',
            'suppliers.*.supplier_sku' => 'nullable|max:255',
            'suppliers.*.leadtime' => 'nullable|integer',
            'suppliers.*.minimum_order_quantity' => 'nullable|numeric|lt:1000000',
            'suppliers.*.operation' => 'nullable|in:'.implode(',', $operationsForObject),
            'suppliers.*.pricing' => 'nullable|array',
            'suppliers.*.pricing.*.supplier_pricing_tier_id' => 'required_without:suppliers.*.pricing.*.supplier_pricing_tier_name|exists:supplier_pricing_tiers,id',
            'suppliers.*.pricing.*.supplier_pricing_tier_name' => 'required_without:suppliers.*.pricing.*.supplier_pricing_tier_id|max:255',
            'suppliers.*.pricing.*.price' => 'nullable|numeric|lt:100000',
            'suppliers.*.pricing.*.operation' => 'nullable|in:'.implode(',', $operationsForObject),

            /**
             * Taxonomy.
             */
            'categories' => 'nullable|array',
            // we can't use category name because of the category name is unique on same category children only
            'categories.*.category_id' => 'required_without:categories.*.category_path',
            'categories.*.category_path' => 'required_without:categories.*.category_id',
            'categories.*.is_primary' => 'nullable|boolean',
            'categories.*.operation' => 'nullable|in:'.implode(',', $operationsForObject),
            //      'attribute_groups'           => 'nullable|array',
            //      'attribute_groups.*'         => 'exists:attribute_groups,id',
            //      'attribute_groups_operation' => 'nullable|in:' . implode( ',', $operationsForArray ),
            'attributes' => 'nullable|array',
            'attributes.*.id' => 'required_without:attributes.*.name|exists:attributes,id',
            'attributes.*.name' => 'required_without:attributes.*.id|max:255',
            'attributes.*.value' => 'nullable',
            'attributes.*.attribute_group_id' => 'nullable|exists:attribute_groups,id',
            'attributes.*.operation' => 'nullable|in:'.implode(',', $operationsForObject),

            /**
             * Variations.
             */
            'variations' => 'array',
            'variations.*.id' => 'required_without:variations.*.sku|exists:products,id',
            'variations.*.sku' => 'required_without:variations.*.id|max:255|unique:products,sku',
            'variations.*.images' => 'array',
            'variations.*.images.*.url' => 'required',
            'variations.*.images.*.name' => 'nullable|max:255',
            'variations.*.images.*.sort_order' => 'nullable|integer',
            'variations.*.images.*.download' => 'nullable|boolean',
            'variations.*.images.*.is_primary' => 'nullable|boolean',
            'variations.*.attributes' => 'nullable|array',
            'variations.*.attributes.*.id' => 'required_without:variations.*.attributes.*.name|exists:attributes,id',
            'variations.*.attributes.*.name' => 'required_without:variations.*.attributes.*.id|max:255',
            //      'variations.*.attributes.*.type'    => 'in:' . implode( ',', Attribute::TYPES ),
            //      'variations.*.attributes.*.variant' => 'boolean',
            'variations.*.attributes.*.value' => 'nullable',

            'blemished' => 'nullable|array',
            'blemished.condition' => 'nullable',
            'blemished.reference' => 'nullable',
            'shared_children_attributes' => 'array',
            'shared_children_attributes.*' => 'exists:attributes,id',
            'is_dropshippable' => 'nullable',
        ];

        /**
         * If Update product we will enable nullable values then filter null values before using it.
         */
        if ($this->isMethod('PUT')) {
            $rules['sku'] = 'unique:products,sku,'.$this->route('product');
            $rules['type'] = 'nullable|in:'.implode(',', Product::TYPES);
            $rules['variations.*.id'] = 'required_without:variations.*.sku|not_in:'.$this->route('product').'|exists:products,id';
            $rules['variations.*.sku'] = 'required_without:variations.*.id|max:255|unique:products,sku,'.$this->route('product').',parent_id';
            $rules['images.*.id'] = [
                'required_if:images.*.operation,'.Helpers::OPERATION_DELETE,
                Rule::exists('product_images', 'id')->where('product_id', $this->route('product')),
            ];
            $rules['images.*.url'] = 'required_without:images.*.id';
            $rules['images.*.operation'] = 'nullable|in:'.implode(',', $operationsForObject);

            // Initial inventory count
            $rules['initial_inventory'] = 'nullable|array';
            $rules['initial_inventory.warehouses'] = 'required_with:initial_inventory|array';
            $rules['initial_inventory.warehouses.*.id'] = 'required_with:initial_inventory.warehouses|exists:warehouses,id';
            $rules['initial_inventory.warehouses.*.quantity'] = 'required_with:initial_inventory.warehouses|numeric|min:0';
            $rules['initial_inventory.warehouses.0.unit_cost'] = 'required_with:initial_inventory.warehouses|numeric|min:0';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (\App\Validator $validator) {
                $attributes = $validator->attributes();

                /**
                 * We will prevent the variations on a variation product.
                 *
                 * Check if the parent product has another parent, if yes , we will prevent that.
                 */
                if (! empty($attributes['parent_id'])) {
                    $parentProduct = Product::with([])->findOrFail($attributes['parent_id']);
                    if (! empty($parentProduct->parent_id)) {
                        $validator->addFailure('parent_id', 'MustNotBeVariation');
                    }
                }

                /**
                 * Check if the product has parent and has variations, we will prevent that.
                 */
                if (! empty($attributes['parent_id']) and ! empty($attributes['variations'])) {
                    $validator->addFailure('variations', 'CannotAddToVariation');
                }

                if (! empty($attributes['variations'])) {
                    $sku = array_column($attributes['variations'], 'sku');
                    $sku[] = $attributes['sku'] ?? '';
                    $sku = array_filter($sku);

                    /**
                     * Check unique sku for each variations and parent.
                     */
                    if (count($sku) != count(array_unique($sku))) {
                        $validator->addFailure('sku', 'Unique');
                    }

                    foreach ($attributes['variations'] as $index => $variation) {
                        /**
                         * Check if variation id equal of parent id.
                         */
                        if (! empty($variation['id']) and ! empty($attributes['parent_id']) and $variation['id'] == $attributes['parent_id']) {
                            $validator->addFailure('variations.'.$index.'.id', 'MustBeDifferent');
                        }

                        /**
                         * Check if variation has at least one variant attribute.
                         */
                        //            $variant_attributes = array_filter( array_column( $variation['attributes'] ?? [], 'variant' ) );
                        //            if ( empty( $variant_attributes ) )
                        //            {
                        //              $validator->addFailure( 'variations.' . $index . '.attributes', 'MustHaveVariantAttribute' );
                        //            }
                    }
                }

                /**
                 * Check if supplier_sku is unique on supplier.
                 */
                if (! empty($attributes['suppliers'])) {
                    if (count($supplierNames = array_column($attributes['suppliers'], 'supplier_name'))) {
                        $suppliers = Supplier::with([])->whereIn('name', $supplierNames)->get(['id', 'name']);
                    }
                    foreach ($attributes['suppliers'] as $index => $supplierProduct) {
                        $supplierId = $supplierProduct['supplier_id'] ?? $suppliers->firstWhere('name', $supplierProduct['supplier_name'])->id;

                        if (isset($supplierProduct['supplier_sku'])) {
                            $supplierProductExists = SupplierProduct::with([])
                                ->where('supplier_sku', $supplierProduct['supplier_sku'])
                                ->where('supplier_id', $supplierId)
                                ->where('product_id', '!=', $this->route('product'))
                                ->exists();
                            if ($supplierProductExists) {
                                $validator->addFailure("suppliers.$index.supplier_sku", 'MustBeUniqueBySupplier');
                            }
                        }
                    }
                }

                // check categories exists and leaf
                if (! empty($attributes['categories'])) {
                    foreach ($attributes['categories'] as $index => $category) {
                        if (! empty($category['category_id'])) {
                            $category = ProductCategory::with([])->find($category['category_id']);
                            $validationAttribute = "categories.$index.category_id";
                        } else {
                            $categoryPath = str_replace(' ', '', $category['category_path']);
                            $category = ProductCategory::with([])->whereRaw("REPLACE(`path`, ' ', '') = '{$categoryPath}'")->first();

                            $validationAttribute = "categories.$index.category_path";
                        }

                        if (is_null($category)) {
                            $validator->addFailure($validationAttribute, 'Exists', []);
                        } elseif (! $category->is_leaf) {
                            $validator->addFailure($validationAttribute, 'IsNonLeaf', []);
                        }
                    }
                }

                // Initial inventory must not be for supplier warehouse
                if (! empty($attributes['initial_inventory'])) {
                    $warehouses = $attributes['initial_inventory']['warehouses'];
                    foreach ($warehouses as $key => $warehouseInfo) {
                        // Make sure warehouse isn't a supplier warehouse.
                        $warehouse = Warehouse::with([])->findOrFail($warehouseInfo['id']);
                        if ($warehouse->isSupplierWarehouse()) {
                            $validator->addFailure('initial_inventory.warehouses.'.$key.'.id', 'MustNotBeSupplierWarehouse');
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_non_leaf' => __('messages.category.non-leaf_failed'),
            'must_be_unique_by_supplier' => __('messages.supplier.supplier_sku_exists'),
            'must_not_be_variation' => __('messages.product.parent_is_variation'),
            'cannot_add_to_variation' => __('messages.product.variation_to_variation'),
            'must_be_different' => __('messages.product.different_id'),
            'must_have_variant_attribute' => __('messages.product.variant_attribute'),
            'must_not_be_supplier_warehouse' => __('messages.warehouse.non_supplier_warehouse'),
        ];
    }

    /**
     * Get custom attributes for validator errors.
     */
    public function attributes(): array
    {
        return [
            'type' => 'product type',
            'name' => 'product name',
            'images.*.url' => 'image url',
            'images.*.name' => 'image name',
            'pricing.*.price' => 'product price',
            'suppliers.*.pricing.*.price' => 'supplier price',
            'attributes.*.id' => 'attribute id',
            'attributes.*.name' => 'attribute name',
            'attributes.*.value' => 'attribute value',
            'variations.*.id' => 'variation id',
            'variations.*.sku' => 'variation sku',
            'variations.*.images' => 'variation images',
            'variations.*.images.*.url' => 'variation image url',
            'variations.*.images.*.name' => 'variation image name',
            'variations.*.images.*.sort_order' => 'variation image sort order',
            'variations.*.images.*.download' => 'variation image download',
            'variations.*.images.*.is_primary' => 'variation image is primary',
            'variations.*.attributes' => 'variation attributes',
            'variations.*.attributes.*.name' => 'variation attribute name',
            'variations.*.attributes.*.value' => 'variation attribute value',
        ];
    }
}
