<?php

namespace App\Http\Requests;

use App\Models\Product;
use App\Models\Warehouse;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class StoreInventoryAdjustment extends FormRequest
{
    use InstanceFromCustom;

    const ADJUSTMENT_TYPE_INCREASE = 'increase';

    const ADJUSTMENT_TYPE_DECREASE = 'decrease';

    const ADJUSTMENT_TYPE_SET = 'set';

    const ADJUSTMENT_TYPES = [
        self::ADJUSTMENT_TYPE_INCREASE,
        self::ADJUSTMENT_TYPE_DECREASE,
        self::ADJUSTMENT_TYPE_SET,
    ];

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'adjustment_date' => 'required|date',
            'product_id' => 'required|exists:products,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'warehouse_location_id' => 'nullable|exists:warehouse_locations,id',
            'quantity' => 'required|numeric|lt:1000000',
            'unit_cost' => 'nullable|numeric|gt:0',
            'notes' => 'nullable',
            'adjustment_type' => 'required|in:'.implode(',', self::ADJUSTMENT_TYPES),
        ];

        if ($this->isMethod('put')) {
            $rules['adjustment_date'] = 'sometimes|'.$rules['adjustment_date'];
            $rules['warehouse_id'] = 'sometimes|'.$rules['warehouse_id'];
            $rules['quantity'] = 'sometimes|'.$rules['quantity'];
            $rules['adjustment_type'] = 'sometimes|in:'.implode(',', [self::ADJUSTMENT_TYPE_DECREASE, self::ADJUSTMENT_TYPE_INCREASE]);
            unset($rules['product_id']); // Product cannot be changed for the adjustment
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (\App\Validator $validator) {
                $attributes = $validator->attributes();

                if ($this->isMethod('PUT')) {
                    // Updating the adjustment
                    $this->validateExtraForUpdate($validator, $attributes);
                } else {
                    // Creating adjustments
                    $this->validateExtraForPost($validator, $attributes);
                }
            });
        }
    }

    private function productHasAverageCost($productId): bool
    {
        $product = Product::with([])->findOrFail($productId);

        return (bool) $product->average_cost;
    }

    private function warehouseHasDefaultLocation($warehouseId): bool
    {
        $warehouse = Warehouse::with(['defaultLocation'])->findOrFail($warehouseId);

        return ! empty($warehouse->defaultLocation);
    }

    private function validateExtraForUpdate(Validator $validator, array $data)
    {
        // If changing the warehouse, it must have a default location
        if (isset($data['warehouse_id']) && ! $this->warehouseHasDefaultLocation($data['warehouse_id'])) {
            $validator->addFailure('warehouse_id', 'NeedsDefaultLocation');
        }
    }

    private function validateExtraForPost(Validator $validator, array $adjustment)
    {
        // Check for non-zero quantity for increase/decrease
        if (in_array($adjustment['adjustment_type'], [
            self::ADJUSTMENT_TYPE_DECREASE,
            self::ADJUSTMENT_TYPE_INCREASE,
        ]) && $adjustment['quantity'] == 0) {
            $validator->addFailure('quantity', 'InvalidQuantity');
        }

        // determined if unit_cost is required
        if (in_array($adjustment['adjustment_type'], [self::ADJUSTMENT_TYPE_INCREASE, self::ADJUSTMENT_TYPE_SET]) &&
         ! isset($adjustment['unit_cost'])) {
            if (! $this->productHasAverageCost($adjustment['product_id'])) {
                $validator->addFailure('unit_cost', 'Required');
            }
        }

        if (empty($adjustment['warehouse_location_id'])) {
            if (! $this->warehouseHasDefaultLocation($adjustment['warehouse_id'])) {
                $validator->addFailure('warehouse_id', 'NeedsDefaultLocation');
            }
        }
    }

    public function messages(): array
    {
        return [
            'needs_default_location' => __('messages.warehouse.warehouse_needs_location'),
            'invalid_quantity' => __('messages.inventory.adjustment_quantity_not_zero'),
        ];
    }

    public function mergeToData($data)
    {
        $this->merge($data);

        $this->getValidatorInstance()->addToData($data);
    }
}
