<?php

namespace App\Http\Requests;

use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class StoreIntegrationInstance extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $defaultTimezoneSetting = Setting::where('key', 'default_timezone')->first()->value;
        $inventoryStartDate = Setting::where('key', 'inventory_start_date')->first()->value->copy()->setTimezone($defaultTimezoneSetting);

        $rules = [
            'integration_id' => 'required|exists:integrations,id',
            'name' => 'required_without:integration_settings.settings.connectionName|unique:integration_instances,name,'.$this->route()->originalParameter('integration_instance'),
            'connection_settings' => 'sometimes|array',
            'integration_settings' => 'sometimes|array',
            'integration_settings.*' => 'nullable', // this rule to read all options
            'integration_settings.settings.connectionName' => 'required_without:name|unique:integration_instances,name,'.$this->route()->originalParameter('integration_instance'),
            'integration_settings.fulfillment.automatedWarehousesIds' => 'nullable|array|not_in:'.implode(',', array_keys(Warehouse::getAutomatedWarehouses())),
            'integration_settings.orders.download.open_start_date' => 'sometimes|date',
            'integration_settings.orders.download.closed_start_date' => 'sometimes|date',
            'integration_settings.proforma_marketplace_cost_percentage' => 'nullable|numeric',
            'integration_settings.proforma_payment_cost_percentage' => 'nullable|numeric',
            'is_automatic_sync_enabled' => 'sometimes|boolean',
            'integration_settings.sync_sales_order_invoices_to_accounting' => 'sometimes|boolean',
        ];

        if ($this->isMethod('PUT')) {
            unset($rules['integration_id'], $rules['integration_settings.fulfillment.automatedWarehousesIds']);

            $rules['name'] = 'sometimes|'.$rules['name'];
            $rules['connection_settings'] = 'sometimes|'.$rules['connection_settings'];
            $rules['integration_settings.settings.connectionName'] = 'sometimes|'.$rules['integration_settings.settings.connectionName'];

            $rules['integration_settings.orders.download.open_start_date'] = 'sometimes|date';
            $rules['integration_settings.orders.download.closed_start_date'] = 'sometimes|date';

            $integrationInstance = request()->route('integration_instance');

            $rules['is_automatic_sync_enabled'] = 'sometimes|boolean';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // validate additional rules based on integration type
                if ($this->isMethod('PUT')) {
                    $integration = $this->route('integration_instance')->integration;
                } else {
                    /** @var Integration $integration */
                    $integration = Integration::with([])->findOrFail($attributes['integration_id']);

                    // check if the integration supports multiple instances
                    if (! $integration->supports_multiple_instances && IntegrationInstance::query()->where('integration_id', $integration->id)->exists()) {
                        $validator->addToData(['integration' => $integration->name]);
                        $validator->addFailure('integration', 'IntegrationDoesNotSupportMultipleInstances');
                    }
                }

                // validate connection settings rules
                if (! request()->has('is_sp_api')) {
                    $validator->validateRules($integration->getConnectionSettingsRules($this->isMethod('PUT')));
                }

                // Get extra rules for integration types
                $integrationTypeRules = $integration->getIntegrationTypeRules($this->isMethod('PUT'));

                // validate integration type rules
                $validator->validateRules($integrationTypeRules);

                if ($this->isMethod('PUT')) {
                    // check automated warehouses belong to other shipping providers
                    if (! empty($attributes['integration_settings']['fulfillment']['automatedWarehousesIds'])) {
                        if (collect(Warehouse::getAutomatedWarehouses())->where('integration_name', '!=', $integration->name)->pluck('warehouse_id')->intersect($attributes['integration_settings']['fulfillment']['automatedWarehousesIds'])->isNotEmpty()) {
                            $validator->addFailure('integration_settings.fulfillment.automatedWarehousesIds', 'NotIn');
                        }
                    }
                }

                // Veracore instance must have an immutable 3pl warehouse
                if($integration->name === Integration::NAME_VERACORE){
                    if($this->isMethod('POST')){
                        if(empty(@$attributes['integration_settings']['linked_warehouse_id'])){
                            $validator->addFailure('linked_warehouse', 'VeracoreMustHaveLinkedWarehouse');
                        } else{
                            /** @var Warehouse $warehouse */
                            $warehouse = Warehouse::query()->findOrFail($attributes['integration_settings']['linked_warehouse_id']);
                            if(!in_array($warehouse->type, [Warehouse::TYPE_3PL, Warehouse::TYPE_DIRECT])){
                                $validator->addFailure('linked_warehouse', 'VeracoreMustHaveLinkedWarehouse');
                            }
                        }
                    } else if($this->isMethod('PUT')){
                        if(isset($attributes['integration_settings']) && empty(@$attributes['integration_settings']['linked_warehouse_id'])){
                            $validator->addFailure('linked_warehouse', 'VeracoreMustHaveLinkedWarehouse');
                        } else if(!empty(@$attributes['integration_settings']['linked_warehouse_id'])) {
                            /** @var Warehouse $warehouse */
                            $warehouse = Warehouse::query()->findOrFail($attributes['integration_settings']['linked_warehouse_id']);
                            if(!in_array($warehouse->type, [Warehouse::TYPE_3PL, Warehouse::TYPE_DIRECT])){
                                $validator->addFailure('linked_warehouse', 'VeracoreMustHaveLinkedWarehouse');
                            }

                            /** @var IntegrationInstance $integrationInstance */
                            $integrationInstance = $this->route('integration_instance');
                            if($integrationInstance->integration_settings['linked_warehouse_id'] != $attributes['integration_settings']['linked_warehouse_id']){
                                $validator->addFailure('linked_warehouse', 'VeracoreLinkedWarehouseCannotBeChanged');
                            }
                        }
                    }
                }

            });
        }
    }

    public function messages(): array
    {
        return [
            'integration_does_not_support_multiple_instances' => __('messages.integration_instance.integration_does_not_support_multiple_instances'),
            'veracore_must_have_linked_warehouse' => 'Veracore instance must be linked to a direct or 3pl warehouse.',
            'veracore_linked_warehouse_cannot_be_changed' => 'Cannot change linked veracore warehouse.',
        ];
    }
}
