<?php

namespace App\Http\Requests;

use App\Models\Address;
use App\Validator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\FormRequest;

class StoreCustomer extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'name' => 'required|max:255',
            'company' => 'nullable|max:255',
            'email' => 'nullable|email|max:255|unique:customers,email,'.$this->route('customer'),
            'phone' => 'nullable',
            'fax' => 'nullable|max:255',
            'address1' => 'nullable|max:255',
            'address2' => 'nullable|max:255',
            'address3' => 'nullable|max:255',
            'city' => 'nullable|max:255',
            'province' => 'nullable|max:255',
            'province_code' => 'nullable|max:255',
            'zip' => 'nullable|max:255',
            'country' => 'nullable|max:255',
            'country_code' => 'nullable|max:2',
            'default_billing_address' => 'nullable|array',
            'default_billing_address.name' => 'nullable|max:255',
            'default_billing_address.company' => 'nullable|max:255',
            'default_billing_address.email' => 'nullable|email|max:255|unique:customers,email,'.$this->route('customer'),
            'default_billing_address.phone' => 'nullable|digits_between:9,13',
            'default_billing_address.fax' => 'nullable|max:255',
            'default_billing_address.address1' => 'nullable|max:255',
            'default_billing_address.address2' => 'nullable|max:255',
            'default_billing_address.address3' => 'nullable|max:255',
            'default_billing_address.city' => 'nullable|max:255',
            'default_billing_address.province' => 'nullable|max:255',
            'default_billing_address.province_code' => 'nullable|max:255',
            'default_billing_address.zip' => 'nullable|max:255',
            'default_billing_address.country' => 'nullable|max:255',
            'default_billing_address.country_code' => 'nullable|max:2',
        ];

        if ($this->isMethod('PUT')) {
            $rules['name'] = 'sometimes|'.$rules['name'];

            // default settings
            $rules['product_pricing_tier_id'] = 'nullable|exists:product_pricing_tiers,id';
            $rules['store_id'] = 'nullable|exists:stores,id';
            $rules['shipping_method_domestic_id'] = 'nullable|exists:shipping_methods,id';
            $rules['shipping_method_international_id'] = 'nullable|exists:shipping_methods,id';
            $rules['warehouse_id'] = 'nullable|exists:warehouses,id';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check email is unique in customer addresses

                if (! empty($attributes['email'])) {
                    if ($this->isMethod('POST')) {
                        $emailExists = Address::with([])
                            ->where('email', $attributes['email'])
                            ->where(function (Builder $builder) {
                                $builder->whereHas('customerByShippingAddress')
                                    ->orWhereHas('customerByBillingAddress')
                                    ->orWhereHas('customerByOtherAddresses');
                            })->exists();
                    } else {
                        $emailExists = Address::with([])
                            ->where('email', $attributes['email'])
                            ->where(function (Builder $builder) {
                                $customerId = $this->route('customer');

                                $builder
                                    ->whereHas('customerByShippingAddress', function (Builder $builder) use ($customerId) {
                                        $builder->where('customers.id', '!=', $customerId);
                                    })
                                    ->orWhereHas('customerByBillingAddress', function (Builder $builder) use ($customerId) {
                                        $builder->where('customers.id', '!=', $customerId);
                                    })
                                    ->orWhereHas('customerByOtherAddresses', function (Builder $builder) use ($customerId) {
                                        $builder->where('customers.id', '!=', $customerId);
                                    });
                            })->exists();
                    }

                    if ($emailExists) {
                        $validator->addFailure('email', 'EmailExistsInCustomerAddresses');
                    }
                }

                if (! empty($attributes['default_billing_address.email'])) {
                    $emailExists = Address::with([])
                        ->where('email', $attributes['default_billing_address.email'])
                        ->where(function (Builder $builder) {
                            $builder->whereHas('customerByShippingAddress')
                                ->orWhereHas('customerByBillingAddress')
                                ->orWhereHas('customerByOtherAddresses');
                        })->exists();

                    if ($emailExists) {
                        $validator->addFailure('default_billing_address.email', 'EmailExistsInCustomerAddresses');
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'email_exists_in_customer_addresses' => __('messages.customer.email_exists'),
        ];
    }
}
