<?php

namespace App\Http\Requests;

use App\Models\Payment;
use App\Models\SalesOrder;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class SalesOrderPaymentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'payment_date' => 'nullable|date',
            'payment_type_id' => 'required|exists:payment_types,id',
            'external_reference' => 'nullable|max:255',
            'amount' => 'required|numeric|lt:100000000',
            'currency_id' => 'nullable|exists:currencies,id',
        ];

        if ($this->isMethod('put')) {
            $rules['payment_type_id'] = 'sometimes|'.$rules['payment_type_id'];
            $rules['amount'] = 'sometimes|'.$rules['amount'];
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // Ensure that the sales order isn't totally paid
                $salesOrder = SalesOrder::with(['salesOrderLines'])->findOrFail(e($this->route('salesOrderId')));

                if ($this->isMethod('post')) {
                    // Creating sales order payment
                    if ($salesOrder->is_fully_paid && $attributes['amount'] > 0) {
                        $validator->addFailure('amount', 'IsFullyPaid');
                    } elseif (isset($attributes['amount']) && $attributes['amount'] < 0) {
                        // Refunds. Refund amount shouldn't be more than what's already paid
                        if (abs($attributes['amount']) > (float) $salesOrder->total_paid) {
                            $validator->addFailure('amount', 'RefundExceedsTotalPaid');
                        }
                    }
                } elseif ($this->isMethod('put')) {
                    // Check that when editing a payment, the payment belongs to the sales order
                    $paymentId = e($this->route('payment'));
                    $payment = $salesOrder->payments()->where('id', $paymentId)->first();
                    if (! $payment) {
                        $validator->addFailure('payment_id', 'PaymentDoesNotBelong');
                    } elseif (isset($attributes['amount']) &&
                    (float) $attributes['amount'] > 0
                    && (float) $attributes['amount'] > (float) (($salesOrder->calculated_total)
                    + ($salesOrder->tax_total ? $salesOrder->tax_total : 0)
                    - $salesOrder->total_paid
                    + $payment->amount)) {
                        $validator->addFailure('amount', 'MoreThanRemaining');
                    } elseif (isset($attributes['amount']) && $attributes['amount'] < 0) {
                        // Refunds. Refund amount shouldn't be more than what's already paid
                        if (abs($attributes['amount']) > (float) $salesOrder->total_paid) {
                            $validator->addFailure('amount', 'RefundExceedsTotalPaid');
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_fully_paid' => __('messages.sales_order.is_fully_paid'),
            'more_than_remaining' => __('messages.sales_order.more_than_remaining'),
            'is_not_approved' => __('messages.sales_order.is_not_approved'),
            'payment_does_not_belong' => __('messages.sales_order.payment_does_not_belong'),
            'refund_exceeds_total_paid' => __('messages.sales_order.refund_exceeds_total_paid'),
        ];
    }
}
