<?php

namespace App\Http\Requests;

use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturnLine;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class ReceiveSalesCreditRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'sales_credit_id' => 'required|exists:sales_credits,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'tracking_number' => 'nullable',
            'shipped_at' => 'nullable|date',
            'received_at' => 'nullable|date',

            // lines

            'return_lines' => 'required|array|min:1',
            'return_lines.*.sales_credit_line_id' => 'required|exists:sales_credit_lines,id',
            'return_lines.*.quantity' => 'required|numeric|gt:0',
            'return_lines.*.action' => 'required|in:'.implode(',', SalesCreditReturnLine::ACTIONS),
            'return_lines.*.reason_id' => 'required|exists:return_reasons,id',
            'return_lines.*.notes' => 'nullable',
            'return_lines.*.blemished_condition' => 'max:255|required_if:return_lines.*.action,'.SalesCreditReturnLine::ACTION_AS_BLEMISHED,
            'return_lines.*.blemished_reference' => 'max:255|nullable',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check sales order lines belong to the sales order and unreturned
                if (! empty($attributes['return_lines'])) {
                    $salesCreditLines = SalesCreditLine::with(['salesCreditReturnLines'])
                        ->whereIn('id', array_column($attributes['return_lines'], 'sales_credit_line_id'))
                        ->get();

                    foreach ($attributes['return_lines'] as $index => $returnLine) {
                        /** @var SalesCreditLine $salesCreditLine */
                        $salesCreditLine = $salesCreditLines->firstWhere('id', $returnLine['sales_credit_line_id']);
                        if ($salesCreditLine->is_product == 0) {
                            throw new \Exception($this->messages()['is_not_product']);
                        }
                        if ($salesCreditLine->fully_received) {
                            $validator->addFailure("return_lines.{$index}.sales_credit_line_id", 'IsReceived');
                        } elseif ($returnLine['quantity'] > $salesCreditLine->unreceived_quantity) {
                            $validator->addFailure("return_lines.{$index}.quantity", 'IsGreaterThanUnreceivedQuantity');
                        }
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_product' => __('messages.sales_order.sales_credit.is_not_product'),
            'is_received' => __('messages.sales_order.sales_credit.fully_received'),
            'is_greater_than_unreceived_quantity' => __('messages.sales_order.sales_credit.greater_than_unreceived'),
        ];
    }
}
