<?php

namespace App\Http\Requests;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Validator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\FormRequest;

class ReceiptPurchaseOrderRequest extends FormRequest
{
    use InstanceFromCustom;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'purchase_order_id' => 'required_without_all:purchase_order_shipment_id,receipt_lines|exists:purchase_orders,id',
            'purchase_order_shipment_id' => 'nullable|exists:purchase_order_shipments,id',
            'received_at' => 'required|date',
            'warehouse_id' => 'nullable|exists:warehouses,id',

            // These are needed for when the user decides to receive all lines,
            // which means we need to first complete any shipments pending.
            'shipment_date' => 'nullable|date',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'tracking' => 'nullable|max:255',

            // lines
            'receipt_lines' => 'nullable|array',
            'receipt_lines.*.purchase_order_shipment_line_id' => 'required_without_all:receipt_lines.*.purchase_order_line_reference,receipt_lines.*.purchase_order_line_id|exists:purchase_order_shipment_lines,id',
            'receipt_lines.*.quantity' => 'required|numeric',
            'receipt_lines.*.purchase_order_line_reference' => 'required_without_all:receipt_lines.*.purchase_order_shipment_line_id,receipt_lines.*.purchase_order_line_id|exists:purchase_order_lines,line_reference',
            'receipt_lines.*.purchase_order_line_id' => 'required_without_all:receipt_lines.*.purchase_order_shipment_line_id,receipt_lines.*.purchase_order_line_reference|exists:purchase_order_lines,id',

            // Unexpected lines
            'unexpected_items' => 'nullable|array',
            'unexpected_items.*.product_id' => 'required_without:unexpected_items.*.product_sku|exists:products,id',
            'unexpected_items.*.product_sku' => 'required_without:unexpected_items.*.product_id|exists:products,sku',
            'unexpected_items.*.quantity' => 'required|numeric',
            'unexpected_items.*.action' => 'nullable',
            'unexpected_items.*.amount' => 'nullable|numeric',
            'unexpected_items.*.description' => 'nullable|string',
        ];

        if ($this->isMethod(static::METHOD_PUT)) {
            $rules = [
                'received_at' => 'required|date',
                'warehouse_id' => 'nullable|exists:warehouses,id',

                // These are needed for when the user decides to receive all lines,
                // which means we need to first complete any shipments pending.
                'shipment_date' => 'nullable|date',
                'shipping_method_id' => 'nullable|exists:shipping_methods,id',
                'tracking' => 'nullable|max:255',

                // lines
                'receipt_lines' => 'required|array',
                'receipt_lines.*.purchase_order_shipment_line_id' => 'required_without_all:receipt_lines.*.purchase_order_line_reference,receipt_lines.*.purchase_order_line_id|exists:purchase_order_shipment_lines,id',
                'receipt_lines.*.quantity' => 'required|numeric',
                'receipt_lines.*.purchase_order_line_reference' => 'required_without_all:receipt_lines.*.purchase_order_shipment_line_id,receipt_lines.*.purchase_order_line_id|exists:purchase_order_lines,line_reference',
                'receipt_lines.*.purchase_order_line_id' => 'required_without_all:receipt_lines.*.purchase_order_shipment_line_id,receipt_lines.*.purchase_order_line_reference|exists:purchase_order_lines,id',
                'unexpected_items' => 'nullable|array',
                'unexpected_items.*.product_id' => 'required_without:unexpected_items.*.product_sku|exists:products,id',
                'unexpected_items.*.product_sku' => 'required_without:unexpected_items.*.product_id|exists:products,sku',
                'unexpected_items.*.quantity' => 'required|numeric',
                'unexpected_items.*.action' => 'nullable',
                'unexpected_items.*.amount' => 'nullable|numeric',
                'unexpected_items.*.description' => 'nullable|string',
            ];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                if (! empty($attributes['unexpected_items']) && ! isset($attributes['purchase_order_id'])) {
                    $validator->addFailure('purchase_order_id', 'Purchase order id is required when unexpected items are present.');
                }

                // check purchase order shipment lines belong to the purchase order shipment and unreceived
                if (! empty($attributes['receipt_lines'])) {
                    // Aggregate receipt lines to ensure that not more than
                    // the shipped quantity is received.
                    $attributes['receipt_lines'] = PurchaseOrderShipmentReceipt::aggregateReceiptLines($attributes['receipt_lines']);

                    foreach ($attributes['receipt_lines'] as $index => $receiptLine) {
                        // We get the provided key for the receipt line
                        $key = $this->getProvidedKey($receiptLine);

                        // Received quantity shouldn't be more than quantity on line
                        // THIS IS REMOVED (SKU-2857) as overages are treated as adjustments or add lines on the fly.
                        //            $this->validateLineQuantity( $validator, $receiptLine, $index );

                        // Make sure that shipment line belongs to shipment id
                        if (isset($attributes['purchase_order_shipment_id'])) {
                            $this->validateOrderShipment($validator, $receiptLine, $index, $key, $attributes);
                        }
                    }
                } else {
                    // No receipt lines provided, user is attempting to either receive everything for a shipment or
                    // the entire purchase order.
                    if (isset($attributes['purchase_order_shipment_id'])) {
                        // Receiving every thing for a shipment.
                        $purchaseOrderShipment = PurchaseOrderShipment::with([])->findOrFail($attributes['purchase_order_shipment_id']);
                        if ($purchaseOrderShipment->fully_received) {
                            $validator->addFailure('purchase_order_shipment', 'ShipmentIsFullyReceived');
                        }
                    } else {
                        // Receiving everything for the purchase order itself.
                        $purchaseOrder = PurchaseOrder::with([])->findOrFail($attributes['purchase_order_id']);
                        if ($purchaseOrder->fully_received) {
                            $validator->addFailure('purchase_order', 'PurchaseOrderIsFullyReceived');
                        }
                    }
                }
            });
        }
    }

    private function validateOrderShipment(Validator $validator, array $receiptLine, $index, $key, array $attributes)
    {
        // We have a shipment id, we need to have a match for the provided line id in shipment lines
        $orderShipmentLine = PurchaseOrderShipmentLine::with([])->where('purchase_order_shipment_id', $attributes['purchase_order_shipment_id'])->first();
        if (! $orderShipmentLine) {
            $validator->addFailure("receipt_lines.{$index}.{$key}", 'IsNotBelongsToPurchaseOrderShipment');
        }

        if ($orderShipmentLine->fully_received) {
            $validator->addFailure("receipt_lines.{$index}.{$key}", 'IsReceived');
        }

        if ($receiptLine['quantity'] > $orderShipmentLine->unreceived_quantity) {
            $validator->addFailure("receipt_lines.{$index}.quantity", 'IsGreaterThanUnreceivedQuantity');
        }
    }

    private function validateLineQuantity(Validator $validator, array $receiptLine, $index)
    {
        // Receipt quantity shouldn't exceed quantity on purchase order line
        $orderLine = PurchaseOrderLine::with([])->findOrFail($receiptLine['purchase_order_line_id']);
        $offset = 0;
        if ($this->isMethod(self::METHOD_PUT)) {
            // We're updating the receipt line, we need to adjust for
            // the current quantity received which will be updated.
            $receiptId = $this->route('receipt');
            $offset = PurchaseOrderShipmentReceipt::with(['purchaseOrderShipmentReceiptLines'])
                ->findOrFail($receiptId)
                ->purchaseOrderShipmentReceiptLines()
                ->whereHas('purchaseOrderShipmentLine', function (Builder $builder) use ($receiptLine) {
                    return $builder->where('purchase_order_line_id', $receiptLine['purchase_order_line_id']);
                })
                ->sum('quantity');
        }
        if ($orderLine->quantity - $orderLine->received_quantity + $offset < $receiptLine['quantity']) {
            $validator->addFailure("receipt_lines.{$index}.quantity", 'IsGreaterThanUnreceivedQuantity');
        }
    }

    private function getProvidedKey(array $receiptLine): string
    {
        if (isset($receiptLine['purchase_order_line_reference'])) {
            return 'purchase_order_line_reference';
        } elseif (isset($receiptLine['purchase_order_shipment_line_id'])) {
            return 'purchase_order_shipment_line_id';
        }

        return 'purchase_order_line_id'; // Always added in when we aggregate quantities.
    }

    public function messages(): array
    {
        return [
            'is_not_belongs_to_purchase_order_shipment' => __('messages.purchase_order.shipment.is_not_belongs'),
            'is_received' => __('messages.purchase_order.shipment.fully_received_line'),
            'shipment_is_fully_received' => __('messages.purchase_order.shipment.fully_received'),
            'purchase_order_is_fully_received' => __('messages.purchase_order.fully_received'),
            'is_greater_than_unreceived_quantity' => __('messages.purchase_order.shipment.greater_than_unreceived'),
        ];
    }
}
