<?php

namespace App\Http\Requests;

use App\Models\PurchaseInvoice;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class PurchaseInvoicePaymentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'payment_date' => 'nullable|date',
            'payment_type_id' => 'required|exists:payment_types,id',
            'external_reference' => 'nullable|max:255',
            'amount' => 'required|numeric|lt:100000000',
            'currency_id' => 'nullable|exists:currencies,id',
        ];

        if ($this->isMethod('put')) {
            $rules['payment_type_id'] = 'sometimes|'.$rules['payment_type_id'];
            $rules['amount'] = 'sometimes|'.$rules['amount'];
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // Ensure that the purchase invoice isn't totally paid
                $purchaseInvoice = PurchaseInvoice::with(['purchaseInvoiceLines'])->findOrFail(e($this->route('purchaseInvoiceId')));
                if ($this->isMethod('post')) {
                    // Creating purchase invoice payment
                    if ($purchaseInvoice->status == PurchaseInvoice::STATUS_PAID && $attributes['amount'] > 0) {
                        $validator->addFailure('amount', 'IsFullyPaid');
                    } elseif (isset($attributes['amount']) &&
                        (float) $attributes['amount'] > 0 &&
                        (float) $attributes['amount'] > (float) (($purchaseInvoice->calculated_total)
                            + ($purchaseInvoice->tax_total)
                            - $purchaseInvoice->total_paid)) {
                        $validator->addFailure('amount', 'MoreThanRemaining');
                    } elseif (isset($attributes['amount']) && $attributes['amount'] < 0) {
                        // Refunds. Refund amount shouldn't be more than what's already paid
                        if (abs($attributes['amount']) > (float) $purchaseInvoice->total_paid) {
                            $validator->addFailure('amount', 'RefundExceedsTotalPaid');
                        }
                    }
                } elseif ($this->isMethod('put')) {
                    // Check that when editing a payment, the payment belongs to the purchase invoice
                    $paymentId = e($this->route('payment'));
                    $payment = $purchaseInvoice->payments()->where('id', $paymentId)->first();
                    if (! $payment) {
                        $validator->addFailure('payment_id', 'PaymentDoesNotBelong');
                    } elseif (isset($attributes['amount']) &&
                        (float) $attributes['amount'] > 0
                        && (float) $attributes['amount'] > (float) (($purchaseInvoice->calculated_total)
                            + ($purchaseInvoice->tax_total ? $purchaseInvoice->tax_total : 0)
                            - $purchaseInvoice->total_paid
                            + $payment->amount)) {
                        $validator->addFailure('amount', 'MoreThanRemaining');
                    } elseif (isset($attributes['amount']) && $attributes['amount'] < 0) {
                        // Refunds. Refund amount shouldn't be more than what's already paid
                        if (abs($attributes['amount']) > (float) $purchaseInvoice->total_paid) {
                            $validator->addFailure('amount', 'RefundExceedsTotalPaid');
                        }
                    }
                }
            });
        }
    }
}
