<?php

namespace App\Http\Requests;

use App\Models\Payment;
use App\Models\SalesCredit;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

/**
 * Class PaidSalesCreditRequest.
 */
class PaidSalesCreditRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'sales_credit_id' => 'required|exists:sales_credits,id',
            'payment_date' => 'nullable|date',
            'payment_type_id' => 'required|exists:payment_types,id',
            'external_reference' => 'nullable|max:255',
            'amount' => 'required|numeric|gt:0|not_in:0',
            'currency_id' => 'nullable|exists:currencies,id',
        ];

        if ($this->isMethod('put')) {
            $rules['sales_credit_id'] = 'sometimes|'.$rules['sales_credit_id'];
            $rules['payment_type_id'] = 'sometimes|'.$rules['payment_type_id'];
            $rules['amount'] = 'sometimes|'.$rules['amount'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check the amount paid
                if ($this->isMethod('post')) {
                    $this->checkAmountPaid($validator, $attributes, $attributes['sales_credit_id']);
                } else {
                    // Payment is being updated, we set the current payment value as offset.
                    $payment = Payment::with([])->findOrFail(e($this->route('payment')));
                    $this->checkAmountPaid($validator, $attributes, $this->route('salesCredit'), -$payment->amount);
                }
            });
        }
    }

    private function checkAmountPaid(Validator $validator, array $attributes, $salesCreditId, $offset = 0)
    {
        $salesCredit = SalesCredit::with(['payments'])->findOrFail($salesCreditId);
        $remainingAmount = ($salesCredit->total_credit + $salesCredit->tax_total) - $salesCredit->paid_amount + $offset;

        if ($remainingAmount == 0) {
            $validator->addFailure('amount', 'IsPaid');
        } elseif ($attributes['amount'] > $remainingAmount) {
            $diff = $remainingAmount - $offset;
            $validator->addFailure('amount', 'Max', ["{$diff}"]);
        }
    }

    public function messages(): array
    {
        return [
            'is_paid' => __('messages.sales_order.sales_credit.fully_paid'),
        ];
    }
}
