<?php

namespace App\Http\Requests;

use App\Helpers;
use App\Http\Controllers\PurchaseOrderController;
use App\Http\Controllers\SalesOrderController;
use App\Http\Controllers\SupplierInventoryController;
use App\Validator;
use Generator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;
use SplFileObject;

class ImportCSVFileRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'csv_file' => 'required|file',
            'csv_delimiter' => 'nullable|max:2',
            'csv_enclosure' => 'nullable|max:2',
            'override' => 'nullable|boolean',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                // check delimiter and enclosure size is one
                // we check it manually because the spacial characters(like "\n"...)
                if (strlen($this->getDelimiter()) > 1) {
                    $validator->addFailure('csv_delimiter', 'Size', [1]);

                    return;
                }

                if (strlen($this->getEnclosure()) > 1) {
                    $validator->addFailure('csv_enclosure', 'Size', [1]);

                    return;
                }

                // check the header has whole required columns
                $file = $this->getCSVFile()->openFile();
                $file->setFlags(SplFileObject::SKIP_EMPTY | SplFileObject::DROP_NEW_LINE);

                $header = $file->fgetcsv($this->getDelimiter(), $this->getEnclosure());
                if ($header) {
                    $header = Helpers::sanitizeCsvRow(
                        mb_convert_encoding($header, 'UTF-8', 'UTF-8')
                    );

                    if (! empty($diffColumns = array_diff($this->getRequiredColumns(), $header))) {
                        foreach ($diffColumns as $column) {
                            $validator->addFailure("csv_file.{$column}", 'ColumnIsRequired');
                        }
                    }
                } else {
                    $validator->addFailure('csv_file', 'IsEmpty');
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'column_is_required' => 'The :attribute column is required',
            'is_empty' => __('messages.failed.csv_no_records'),
        ];
    }

    /**
     * @return array|string[]
     */
    private function getRequiredColumns()
    {
        if ($this->route()->getController() instanceof SalesOrderController && $this->route()->getActionMethod() == 'importCSV') {
            return [
                'order_status',
                'currency_code',
                'order_date',
                'order_line_description',
                'order_line_quantity',
                'order_line_amount',
                'customer_name',
            ];
        } elseif ($this->route()->getController() instanceof PurchaseOrderController && $this->route()->getActionMethod() == 'importCSV') {
            return [
                'supplier_name',
                'currency_code',
                'order_status',
                'purchase_order_date',
                'order_line_description',
                'order_line_sku',
                'order_line_quantity',
                'order_line_amount',
            ];
        } elseif ($this->route()->getController() instanceof SupplierInventoryController && $this->route()->getActionMethod() === 'importCSV') {
            return [
                'product_sku',
                'warehouse_name',
                'quantity',
                'in_stock',
                'eta',
                'source',
            ];
        }

        return [];
    }

    /**
     * Get CSV file.
     */
    public function getCSVFile(): UploadedFile
    {
        return $this->file('csv_file');
    }

    /**
     * Get the column delimiter.
     */
    public function getDelimiter(): string
    {
        return stripcslashes($this->input('csv_delimiter', ','));
    }

    /**
     * Get the column enclosure character.
     */
    public function getEnclosure(): string
    {
        return stripcslashes($this->input('csv_enclosure', '"'));
    }

    public function overrideData(): bool
    {
        return (bool) $this->input('override', true);
    }

    /**
     * Convert csv content to associated array.
     */
    public function getRows(): Generator
    {
        return Helpers::csvFileToArray($this->getCSVFile()->openFile(), $this->getDelimiter(), $this->getEnclosure());
    }

    /**
     * Get elements from array that start with the specified prefix.
     */
    public function getElementsByPrefix(array $array, string $prefix): array
    {
        $result = [];

        foreach ($array as $key => $value) {
            if (Str::startsWith($key, $prefix)) {
                $result[str_replace($prefix, '', $key)] = $value;
            }
        }

        return $result;
    }
}
