<?php

namespace App\Http\Requests;

use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Response;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

/**
 * Class StoreSalesOrderFulfillment.
 */
class FulfillSalesOrderRequest extends FormRequest
{
    use InstanceFromCustom, LockRequestTrait;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $isSalesOrderParameter = $this->route()->hasParameter('sales_order');
        $rules = [
            'fulfillment_type' => 'required|in:'.implode(',', SalesOrderFulfillment::TYPES),
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'requested_shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'requested_shipping_method' => 'nullable|max:255',
            'fulfilled_shipping_method_id' => 'nullable|exists:shipping_methods,id', // for manual fulfillment
            'fulfilled_shipping_method' => 'nullable|max:255',
            'fulfilled_at' => 'required_if:fulfillment_type,'.SalesOrderFulfillment::TYPE_MANUAL.'|date',
            'cost' => 'nullable|numeric|lt:100000',
            'tracking_number' => 'nullable|max:255',
            'fulfillment_sequence' => ['nullable', 'integer', Rule::unique('sales_order_fulfillments', 'fulfillment_sequence')->where('sales_order_id', $this->route($isSalesOrderParameter ? 'sales_order' : 'sales_order_fulfillment')->id)],
            'fulfillment_lines' => 'array|min:1',
            'fulfillment_lines.*.sales_order_line_id' => 'required|exists:sales_order_lines,id',
            'fulfillment_lines.*.quantity' => 'required|numeric|min:0|lt:100000',
            'fulfillment_lines.*.metadata.fba_seller_sku' => 'sometimes|string',
            'metadata.signature_required' => 'sometimes|bool',
            'metadata.saturday_delivery' => 'sometimes|bool',
            'metadata.shipping_speed' => 'sometimes|string',
            'metadata.comments' => 'sometimes|string',
            'packing_slip_printed_at' => 'nullable|date',
            'status' => 'sometimes|string',
            'submit_to_shipping_provider' => 'sometimes|bool',
            'submit_to_sales_channel' => 'sometimes|bool',
        ];

        if ($this->isMethod('PUT')) {
            unset($rules['warehouse_id']);
            $rules['fulfillment_type'] = 'sometimes|'.$rules['fulfillment_type'];
            $rules['fulfilled_at'] = 'sometimes|'.$rules['fulfilled_at'];
            // send fulfillment line id or sales order id
            $rules['fulfillment_lines.*.id'] = 'required_without:fulfillment_lines.*.sales_order_line_id|exists:sales_order_fulfillment_lines,id';
            $rules['fulfillment_lines.*.sales_order_line_id'] = 'required_without:fulfillment_lines.*.id|exists:sales_order_lines,id';
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                // check sales order is exists and approved
                /** @var SalesOrder $salesOrder */
                $salesOrder = $this->isMethod('POST') ? $this->route('sales_order') : $this->route('sales_order_fulfillment')->salesOrder;
                // sales order status must be open
                $this->checkSalesOrderStatus($salesOrder, $validator);

                // check sales order lines belong to sales order and unfulfilled
                $salesOrder->load('salesOrderLines.backorderQueue', 'salesOrderLines.salesOrderFulfillmentLines');

                if ($this->isMethod('POST')) {
                    $productWarehouses = $salesOrder->salesOrderLines->where('is_product', true)->pluck('warehouse_id')->unique();

                    if ($productWarehouses->count() === 1) {
                        $attributes['warehouse_id'] = $productWarehouses->first();
                        $this->getValidatorInstance()->addToData(['warehouse_id' => $attributes['warehouse_id']]);
                    } else {
                        if (empty($attributes['warehouse_id'])) {
                            $validator->addFailure('warehouse_id', 'Required');
                        } elseif (! $productWarehouses->firstWhere(null, $attributes['warehouse_id'])) {
                            $validator->addFailure('warehouse_id', 'IsNotBelongsToSalesOrderWarehouses');
                        }
                    }

                    if (! empty($attributes['warehouse_id']) && empty($attributes['fulfillment_lines'])) {
                        $unfulfilledLines = [];
                        foreach ($salesOrder->salesOrderLines as $line) {
                            if ($line->warehouse_id == $attributes['warehouse_id'] && $line->is_product && $line->unfulfilled_quantity) {
                                $unfulfilledLines[] = ['sales_order_line_id' => $line->id, 'quantity' => $line->unfulfilled_quantity];
                            }
                        }
                        $attributes['fulfillment_lines'] = $unfulfilledLines;
                        $this->getValidatorInstance()->addToData(['fulfillment_lines' => $attributes['fulfillment_lines']]);
                    }
                }

                foreach ($attributes['fulfillment_lines'] ?? [] as $index => $fulfillmentLine) {
                    /** @var SalesOrderLine $salesOrderLine */
                    $salesOrderLine = $salesOrder->salesOrderLines->firstWhere('id', $fulfillmentLine['sales_order_line_id']);

                    if ($salesOrderLine->sales_order_id != $salesOrder->id) {
                        $validator->addFailure("fulfillment_lines.{$index}.sales_order_line_id", 'IsNotBelongsToSalesOrder');
                    }

                    if (isset($attributes['warehouse_id']) && $salesOrderLine->warehouse_id != $attributes['warehouse_id']) {
                        $validator->addFailure("fulfillment_lines.{$index}.sales_order_line_id", 'IsNotBelongsToWarehouse');
                    }
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_not_approved' => __('messages.sales_order.is_not_approved'),
            'is_closed' => __('messages.sales_order.is_closed'),
            'is_not_belongs_to_sales_order' => __('messages.sales_order.is_not_belongs'),
            'is_not_belongs_to_sales_order_warehouses' => __('messages.sales_order.is_not_belongs_warehouses'),
            'is_not_belongs_to_warehouse' => __('messages.sales_order.is_not_belongs_to_warehouse'),
        ];
    }

    /**
     * Checks if the sales order line is backordered.
     */
    protected function isBackordered(SalesOrderLine $salesOrderLine, int $quantity): bool
    {
        return $quantity > $salesOrderLine->fulfillable_quantity;
    }

    /**
     * Determine if the sales order status is open.
     */
    private function checkSalesOrderStatus(SalesOrder $salesOrder, Validator $validator)
    {
        if ($salesOrder->order_status == SalesOrder::STATUS_DRAFT) {
            $validator->addFailure('sales_order_id', Response::CODE_NOT_APPROVED);
        }

        if ($salesOrder->order_status == SalesOrder::STATUS_CLOSED && $salesOrder->fulfillment_status != SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC) {
            $validator->addFailure('sales_order_id', Response::CODE_CLOSED);
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function lockId(): string
    {
        return ($this->isMethod('POST') ? $this->route('sales_order') : $this->route('sales_order_fulfillment')->salesOrder)->id;
    }
}
