<?php

namespace App\Http\Requests;

use App\Models\ProductCategory;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class CategoryToProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $rules = [
            'category_id' => [
                Rule::unique('product_to_categories', 'category_id')
                    ->where('product_id', $this->input('product_id')),
            ],
            'product_id' => 'exists:products,id',
            'is_primary' => 'nullable|boolean',
        ];

        if ($this->getMethod() === 'POST') {
            // Add required fields
            $rules['category_id'][] = 'required';
            $rules['product_id'] = 'required|'.$rules['product_id'];
        }

        return $rules;
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();

                if ($this->getMethod() === 'PUT' && ! isset($attributes['category_id'])) {
                    return;
                }

                $category = ProductCategory::with([])->find($attributes['category_id']);
                if (is_null($category)) {
                    $validator->addFailure('category_id', 'Exists');
                } elseif (! $category->is_leaf) {
                    $validator->addFailure('category_id', 'IsNonLeaf');
                }
            });
        }
    }

    public function messages(): array
    {
        return [
            'is_non_leaf' => __('messages.category.non-leaf_failed'),
        ];
    }
}
