<?php

namespace App\Http\Requests;

use App\Models\Customer;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Validator;
use Illuminate\Foundation\Http\FormRequest;

class AllocateCustomerCreditToOrdersRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'sales_orders' => 'required|array',
            'sales_orders.*.id' => 'required|exists:sales_orders,id',
            'sales_orders.*.amount' => 'required|numeric|gt:0',
        ];
    }

    public function withValidator(Validator $validator)
    {
        if ($validator->passes()) {
            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();
                // Total amounts should not exceed un-allocated amounts on the sales credit
                $customer = Customer::with(['salesCredits'])->findOrFail(e($this->route('customer')));
                /** @var SalesCredit $salesCredit */
                $salesCredit = $customer->salesCredits()->with(['salesCreditLines'])->findOrFail(e($this->route('salesCredit')));
                if ($salesCredit->total_credit - $salesCredit->paid_amount < $this->totalSalesOrderAllocation($attributes['sales_orders'])) {
                    $validator->addFailure('amount', 'AllocationsExceedTotalAmount');

                    return;
                }

                // Make sure that allocated amount does not exceed unpaid amount for each sales order
                foreach ($attributes['sales_orders'] as $key => $order) {
                    $salesOrder = SalesOrder::with(['payments'])->findOrFail($order['id']);

                    if ($salesOrder->is_fully_paid) {
                        $validator->addFailure("sales_orders.{$key}.{$order['id']}", 'IsFullyPaid');
                    } elseif ($salesOrder->total - $salesOrder->total_paid < $order['amount']) {
                        $validator->addFailure("sales_orders.{$key}.{$order['id']}", 'MoreThanRemaining');
                    }
                }
            });
        }
    }

    private function totalSalesOrderAllocation(array $salesOrders): float
    {
        return collect($salesOrders)->sum('amount');
    }

    public function messages(): array
    {
        return [
            'allocations_exceed_total_amount' => __('messages.sales_order.sales_credit.allocations_exceed_total_amount'),
            'more_than_remaining' => __('messages.sales_order.more_than_remaining'),
            'is_fully_paid' => __('messages.sales_order.is_fully_paid'),
        ];
    }
}
