<?php

namespace App\Http\Controllers;

use App\Data\UpdateWarehouseTransferLinesData;
use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\ReceivingDiscrepanciesMissingNominalCodeMappingException;
use App\Exceptions\ReceivingDiscrepancyAlreadyExistsException;
use App\Exceptions\UsedFifoLayerException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferHasNoProductsException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferOpenException;
use App\Helpers;
use App\Helpers\ExcelHelper;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreWarehouseTransfer;
use App\Http\Resources\AccountingTransactionResource;
use App\Http\Resources\WarehouseTransferResource;
use App\Managers\WarehouseTransferManager;
use App\Models\Product;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Repositories\WarehouseTransferRepository;
use App\Response;
use App\Services\Accounting\AccountingTransactionManager;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Throwable;

class WarehouseTransferController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = WarehouseTransfer::class;

    protected $resource = 'warehouse transfer';

    /**
     * WarehouseTransferController constructor.
     */
    public function __construct(
        private readonly WarehouseTransferManager $transferManager,
        private readonly WarehouseTransferRepository $transferRepository,
    )
    {
        parent::__construct();
    }

    public function show(WarehouseTransfer $transfer): Response
    {
        $transfer->load(array_merge(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class), [
            'notes',
            'warehouseTransferLines.adjustments.product',
            'warehouseTransferLines.shipmentLine.receiptLines',
            'warehouseTransferLines.shipmentLine.amazonFbaLedger',
            'toWarehouse.address',
            'shipments.shipmentLines.warehouseTransferLine.product',
            'shipments.shipmentLines.receiptLines',
            'shipments.shippingMethod',
        ]));

        return $this->response->addData(WarehouseTransferResource::make($transfer));
    }

    /**
     * Transfers between warehouses.
     *
     *
     * @throws Throwable
     */
    public function store(StoreWarehouseTransfer $request): JsonResponse
    {
        set_time_limit(-1);
        try {
            // Perform the transfer
            $transfer = $this->transferManager->initiateTransfer($request->validated());
            $transfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'warehouse transfer']))
                ->addData(WarehouseTransferResource::make($transfer));
        } catch (InsufficientStockException $e) {
            // Some product doesn't have enough fifo layer
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_no_quantity', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }
    }

    /**
     * Duplicates a warehouse transfer.
     *
     *
     * @throws Throwable
     */
    public function duplicate($transferId): JsonResponse
    {
        $transfer = WarehouseTransfer::with([])->findOrFail(e($transferId));

        // Duplicate the transfer
        $transfer = $this->transferManager->duplicate($transfer);
        $transfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => 'warehouse transfer']))
            ->addData(WarehouseTransferResource::make($transfer));
    }

    /**
     * Opens a warehouse transfer.
     */
    public function openWarehouseTransfer(StoreWarehouseTransfer $request, $transferId): Response
    {

        /** @var WarehouseTransfer $warehouseTransfer */
        $warehouseTransfer = WarehouseTransfer::with([])->findOrFail(e($transferId));

        $payload = $request->validated();

        // We open the warehouse transfer
        try {
            $transfer = $this->transferManager->openWarehouseTransfer($warehouseTransfer, $payload);
            $transfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.open', ['resource' => 'warehouse transfer']))
                ->addData(WarehouseTransferResource::make($transfer));
        } catch (WarehouseTransferOpenException $e) {
            return $this->sendTransferAlreadyOpenResponse($e->transfer);
        } catch (InsufficientStockException $e) {
            // Some product doesn't have enough fifo layer
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_no_quantity', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        } catch (WarehouseTransferHasNoProductsException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_no_products'), 'WarehouseTransfer'.Response::CODE_UNACCEPTABLE, 'warehouse transfer');
        }
    }

    public function makeWarehouseTransferDraft($transferId)
    {
        $warehouseTransfer = WarehouseTransfer::with([])->findOrFail(e($transferId));

        try {
            // We open the warehouse transfer
            $transfer = $this->transferManager->makeTransferDraft($warehouseTransfer);
            $transfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.draft', ['resource' => 'warehouse transfer']))
                ->addData(WarehouseTransferResource::make($transfer));
        } catch (UsedFifoLayerException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.product.used_fifo_layer', ['sku' => $e->fifoLayer->product->sku]), 'WarehouseTransfer'.Response::CODE_UNACCEPTABLE, 'warehouse transfer');
        }
    }

    /**
     * Updates a warehouse transfer.
     */
    public function update(StoreWarehouseTransfer $request, $transferId): Response
    {
        /** @var WarehouseTransfer $warehouseTransfer */
        $warehouseTransfer = WarehouseTransfer::with([])->findOrFail(e($transferId));

        // For now, we only allow opening a warehouse transfer
        $payload = $request->validated();

        // We update the warehouse transfer
        try {
            $this->transferManager->updateTransfer($warehouseTransfer, $payload);
            $warehouseTransfer->load(DataTableConfiguration::getRequiredRelations(WarehouseTransfer::class));

            return $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.update', ['resource' => 'warehouse transfer']))
                ->addData(WarehouseTransferResource::make($warehouseTransfer));
        } catch (WarehouseTransferOpenException $e) {
            return $this->sendTransferAlreadyOpenResponse($e->transfer);
        } catch (InsufficientStockException $e) {
            // Some product doesn't have enough fifo layer
            return $this->response->error(Response::HTTP_UNPROCESSABLE_ENTITY)
                ->addError(__('messages.warehouse.transfer_no_quantity', [
                    'extra' => '',
                    'product_id' => $e->productId,
                    'sku' => $e->productSku,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }
    }

    private function sendTransferAlreadyOpenResponse(WarehouseTransfer $transfer): Response
    {
        return $this->response->error(Response::HTTP_BAD_REQUEST)
            ->addError(__('messages.warehouse.transfer_status_open', [
                'transfer_id' => $transfer->id,
                'extra' => '',
            ]), 'WarehouseTransfer'.Response::CODE_IS_ALREADY_OPEN, 'transfer_id', ['transfer_id' => $transfer->id]);
    }

    /**
     * Delete warehouse transfer.
     *
     *
     * @throws Exception
     */
    public function destroy(WarehouseTransfer $transfer): Response
    {
        try {
            // Delete the warehouse transfer
            $transfer->delete();

            return $this->response->setMessage(__('messages.success.delete', [
                'resource' => $this->resource,
                'id' => $transfer->id,
            ]));
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_fifo_used', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }
    }

    /**
     * Bulk delete warehouses.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        try {
            return $this->bulkOperation($request, $this->BULK_DELETE);
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_fifo_used', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:warehouse_transfers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        // All warehouse transfers are deletable,
        // only reverses inventory movements.

        $result = [];
        $transfers = WarehouseTransfer::with(['warehouseTransferLines.product'])->whereIn('id', $ids)->select('id', 'receipt_status', 'warehouse_transfer_number')->get();
        foreach ($transfers as $key => $transfer) {
            $usage = $transfer->isUsed();
            $result[$key] = $transfer->only('id');
            $result[$key]['deletable'] = $usage === false;
            $result[$key]['reason'] = $usage ?: null;
        }

        return $this->response->addData($result);
    }

    public function archive(WarehouseTransfer $transfer): Response
    {
        if ($transfer->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $transfer->id,
                ]))
                ->addData(WarehouseTransferResource::make($transfer));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $transfer->id,
            ]), 'WarehouseTransfer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $transfer->id])
            ->addData(WarehouseTransferResource::make($transfer));
    }

    public function unarchived(WarehouseTransfer $transfer): Response
    {
        if ($transfer->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $transfer->id,
                ]))
                ->addData(WarehouseTransferResource::make($transfer));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $transfer->id,
            ]), 'WarehouseTransfer'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $transfer->id])
            ->addData(WarehouseTransferResource::make($transfer));
    }

    /**
     * Imports Warehouse Transfer Lines from CSV file
     *
     *
     * @throws Throwable
     */
    public function importLinesCsv(Request $request): Response
    {
        $request->validate([
            'file' => 'required_without:csvString',
            'csvString' => 'required_without:file',
            'warehouse_transfer_id' => 'required',
            'replace' => 'nullable|boolean', ]);
        $separator = $request->input('separator', ',');
        $escape = $request->input('escape', '"');
        $syncLines = $request->input('replace', false);
        /** @var WarehouseTransfer $warehouseTransfer */
        $warehouseTransfer = WarehouseTransfer::query()->findOrFail($request->input('warehouse_transfer_id'));

        // csv data form file or from string
        if ($request->input('file')) {
            $filePath = storage_path(config('uploader.models.target').$request->input('file'));
            $csvData = Helpers::csvFileToArray($filePath, $separator, $escape);
            $header = array_keys($csvData->current());
        } else {
            $csvData = Helpers::csvToArray($request->input('csvString'), $separator, $escape);
            $header = array_keys($csvData[0] ?? []);
        }

        // Headers require to be in CSV file
        $headers = ['sku'];
        if (count(array_intersect($header, $headers)) === 0) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError(__('messages.import_export.mismatch_template'), Response::CODE_INVALID_Template, 'file');
        }

        $skuArr = [];
        $unknownSkus = [];
        foreach ($csvData as $line) {
            // exclude rows that have empty sku
            if (empty(trim($line['sku']))) {
                continue;
            }

            if (!$product = Product::query()->where('sku', $line['sku'])->first(['id', 'name', 'sku'])) {
                $unknownSkus[] = $line['sku'];
                continue;
            }

            $skuArr[$line['sku']]['quantity'] = ($skuArr[$line['sku']]['quantity'] ?? 0) + (empty($line['quantity']) ? 1 : $line['quantity']);
            $skuArr[$line['sku']]['id'] = $product?->id;
        }

        if (!empty($unknownSkus)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage('The CSV contains unknown products')
                ->addError('The CSV contains unknown products', 'Product'.Response::CODE_NOT_FOUND, 'unknown_skus', $unknownSkus);
        }

        if (empty($skuArr)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.csv_no_records'));
        }

        // create and validate warehouse transfer update request
        $updateRequest = StoreWarehouseTransfer::createFromCustom(['products' => array_values($skuArr)], 'PUT', ['transfer' => $warehouseTransfer]);
        $inputs = $updateRequest->validated();
        try {
            // add lines to the warehouse transfer
            DB::transaction(function () use ($syncLines, $inputs, $warehouseTransfer) {
                $this->transferManager->updateTransferProducts($warehouseTransfer, $inputs, $syncLines);
            });
        } catch (InsufficientStockException $e) {
            // Some product doesn't have enough fifo layer
            return $this->response->error(Response::HTTP_UNPROCESSABLE_ENTITY)
                ->addError(__('messages.warehouse.transfer_no_quantity', [
                    'extra' => '',
                    'product_id' => $e->productId,
                    'sku' => $e->productSku,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }

        return $this->response->setMessage(__('messages.success.import', ['resource' => 'warehouse transfer lines']));
    }

    /**
     * Download importing lines template.
     */
    public function exportLines($id): BinaryFileResponse
    {
        $lines = collect();

        WarehouseTransferLine::with(['product', 'warehouseTransfer'])
            ->where('warehouse_transfer_id', $id)
            ->each(function (WarehouseTransferLine $warehouseTransferLine) use ($lines) {
                $line = [
                    'warehouse_transfer_number' => $warehouseTransferLine->warehouseTransfer->warehouse_transfer_number,
                    'sku' => $warehouseTransferLine->product->sku,
                    'quantity' => $warehouseTransferLine->quantity,
                ];
                $lines->add($line);
            });

        $headers = ['warehouse_transfer_number', 'sku', 'quantity']; // static to return them when lines are empty

        $exportedFile = ExcelHelper::array2csvfile('sales_order_lines.csv', $headers, $lines);

        return response()->download($exportedFile)->deleteFileAfterSend(true);
    }

    /**
     * @throws Throwable
     */
    public function createReceivingDiscrepancy(WarehouseTransfer $transfer): Response
    {
        try {
            $accountingTransaction = app(AccountingTransactionManager::class)->createReceivingDiscrepancyFromWarehouseTransfer($transfer);
        } catch (ReceivingDiscrepanciesMissingNominalCodeMappingException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'ReceivingDiscrepancyMissingNominalCodeMapping', 'nominal_code_id');
        } catch (ReceivingDiscrepancyAlreadyExistsException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'ReceivingDiscrepancyAlreadyExists', 'purchase_order_id');
        }

        return $this->response->addData(AccountingTransactionResource::make($accountingTransaction));
    }

    public function updateWarehouseTransferLines(WarehouseTransfer $warehouseTransfer, UpdateWarehouseTransferLinesData $data): Response
    {
        $this->transferRepository->updateWarehouseTransferLines($warehouseTransfer, $data);

        return $this->response->addData(WarehouseTransferResource::make($warehouseTransfer));
    }

    protected function getModel(): string
    {
        return WarehouseTransfer::class;
    }

    protected function getResource(): string
    {
        return WarehouseTransferResource::class;
    }
}
