<?php

namespace App\Http\Controllers;

use App\Data\UserData;
use App\DataTable\DataTable;
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Requests\User\StoreUserRequest;
use App\Http\Resources\UserResource;
use App\Managers\UserManager;
use App\Models\User;
use App\Response;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Throwable;

class UserController extends RegisterController
{
    use DataTable;

    public function store(Request $request)
    {
        $this->validator($request->all())->validate();

        event(new Registered($user = $this->create($request->all())));

        if ($response = $this->registered($request, $user)) {
            return $response;
        }

        return $request->wantsJson()
                    ? new JsonResponse([], 201)
                    : redirect($this->redirectPath());
    }

    public function update(StoreUserRequest $request, User $user)
    {
        //Separate check for is_admin because is_admin is not in fillable
        if (isset($request->is_admin) && auth()->user()->is_admin === true) {
            $user->is_admin = $request->is_admin;
        }

        $user->update($request->validated());

        return $this->response->setMessage(__('messages.success.update', ['resource' => 'user', 'id' => $user->name]));
    }

    protected function getModel()
    {
        return User::class;
    }

    protected function getResource()
    {
        return UserResource::class;
    }

    /**
     * Get the user details
     */
    public function show(int $id): Response
    {
        $user = User::findOrFail($id);

        return $this->response->addData(UserResource::make($user));
    }

    //soft delete user
    public function destroy(User $user): Response
    {
        if ($user->trashed()) {
            $user->restore();

            return $this->response->setMessage(__('User activated successfully', ['resource' => 'user', 'id' => $user->name]));
        } else {
            $user->delete();

            return $this->response->setMessage(__('messages.success.delete', ['resource' => 'user', 'id' => $user->name]));
        }
    }

    /**
     * Create User from UI and send email to update password
     *
     * @throws Throwable
     */
    public function createUser(UserData $userData, UserManager $userManager): Response
    {
        $user = $userManager->createUserAndSendInvitation($userData);

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'user', 'id' => $user->name]));
    }
}
