<?php

namespace App\Http\Controllers\Traits;

use App\Models\Address;
use App\Models\SalesOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use Ballen\Distical\Calculator;
use Ballen\Distical\Entities\LatLong;
use Exception;
use Illuminate\Http\JsonResponse;
use InvalidArgumentException;

trait SalesOrderSplitActions
{
    private $productLineItemNominalCode = 1;

    /**
     * Splitting an order by availability(warehouses).
     *
     *
     * @throws Exception
     */
    public function splitByAvailability($salesOrderId): JsonResponse
    {
        $salesOrder = SalesOrder::with([
            'salesOrderLines',
            'salesOrderLines.product',
            'salesOrderLines.product.inWarehousesQuantity',
            'shippingAddress',
        ])->findOrFail($salesOrderId);

        if (! $salesOrder->shippingAddress) {
            return response()->json([
                'status' => __('messages.status_failed'),
                'messages' => __('messages.sales_order.split_availability_missing_shipping_address'),
            ]);
        }

        try {
            if (is_null($shippingAddressCoordinates = $this->getCoordinatesFromAddress($salesOrder->shippingAddress))) {
                return response()->json([
                    'status' => __('messages.status_failed'),
                    'messages' => __('messages.sales_order.split_availability_coordinates_shipping_address'),
                ]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => __('messages.status_failed'), 'messages' => $e->getMessage()]);
        }

        // coordinates of sales order shipping address
        $shippingAddressLatLong = new LatLong($shippingAddressCoordinates['lat'], $shippingAddressCoordinates['lng']);

        $orders = ['available' => [], 'backorder' => []];
        // warehouses and coordinates
        $warehouses = Warehouse::all();
        $warehousesCoordinates = [];

        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            $productWarehouses = [];
            // check the sales order line is a product
            if ($salesOrderLine->nominal_code_id == $this->productLineItemNominalCode && ! empty($salesOrderLine->product_id)) {
                foreach ($salesOrderLine->product->inWarehousesQuantity as $inWarehouseQuantity) {
                    if ($inWarehouseQuantity->quantity >= $salesOrderLine->quantity) {
                        if (! isset($warehousesCoordinates[$inWarehouseQuantity->warehouse_id])) {
                            $warehousesCoordinates[$inWarehouseQuantity->warehouse_id] = $this->getCoordinatesFromAddress($inWarehouseQuantity->warehouse->default_address);

                            if (is_null($warehousesCoordinates[$inWarehouseQuantity->warehouse_id])) {
                                return response()->json([
                                    'status' => __('messages.status_failed'),
                                    'messages' => __('messages.sales_order.split_availability_coordinates_warehouse', ['warehouse' => $inWarehouseQuantity->warehouse->name]),
                                ]);
                            }
                        }

                        if ($warehousesCoordinates[$inWarehouseQuantity->warehouse_id]) {
                            $productWarehouses[$inWarehouseQuantity->warehouse_id] = null;
                        }
                    }
                }

                // check its product has available quantity and warehouses coordinates
                if (count($productWarehouses)) {
                    foreach ($productWarehouses as $productWarehouse => $distance) {
                        $warehouseLatLong = new LatLong($warehousesCoordinates[$productWarehouse]['lat'], $warehousesCoordinates[$productWarehouse]['lng']);

                        // Get the distance between these two Lat/Long coordinates...
                        $distance = new Calculator($shippingAddressLatLong, $warehouseLatLong);

                        try {
                            $productWarehouses[$productWarehouse] = $distance->get()->asKilometres();
                        } catch (InvalidArgumentException $exception) {
                            $productWarehouses[$productWarehouse] = 0;
                        }
                    }

                    // get nearest warehouse by minimum distance
                    $nearestWarehouse = array_search(min($productWarehouses), $productWarehouses);

                    // add order line to the nearest warehouse
                    if (isset($orders['available'][$nearestWarehouse])) {
                        $orders['available'][$nearestWarehouse]['total_weight'] += ($salesOrderLine->product->weight * $salesOrderLine->quantity);
                        $orders['available'][$nearestWarehouse]['lines'][$salesOrderLine->id] = [
                            'id' => $salesOrderLine->id,
                            'quantity' => $salesOrderLine->quantity,
                            'sku' => $salesOrderLine->product->sku,
                            'description' => $salesOrderLine->description,
                        ];
                    } else {
                        $orders['available'][$nearestWarehouse] = [
                            'total_weight' => $salesOrderLine->product->weight * $salesOrderLine->quantity,
                            'lines' => [
                                $salesOrderLine->id => [
                                    'id' => $salesOrderLine->id,
                                    'quantity' => $salesOrderLine->quantity,
                                    'sku' => $salesOrderLine->product->sku,
                                    'description' => $salesOrderLine->description,
                                ],
                            ],
                            'warehouse' => [
                                'id' => $nearestWarehouse,
                                'name' => $warehouses->firstWhere('id', $nearestWarehouse)->name,
                            ],
                        ];
                    }
                } else {
                    if (! isset($orders['backorder']['lines'])) {
                        $orders['backorder']['lines'] = [];
                    }

                    $orders['backorder']['lines'][] = [
                        $salesOrderLine->id => [
                            'id' => $salesOrderLine->id,
                            'quantity' => $salesOrderLine->quantity,
                            'sku' => $salesOrderLine->product->sku,
                            'description' => $salesOrderLine->description,
                        ],
                    ];
                }
            }
        }

        return response()->json($orders);
    }

    /**
     * Splitting an order by Weight.
     */
    public function splitByWeight($salesOrderId, $maxWeight): JsonResponse
    {
        $salesOrder = SalesOrder::with(['salesOrderLines', 'salesOrderLines.product'])->findOrFail($salesOrderId);

        $index = 1;
        $orders = [$index => ['total_weight' => 0, 'lines' => []]];

        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            // check the sales order line is a product
            if ($salesOrderLine->nominal_code_id == $this->productLineItemNominalCode && ! empty($salesOrderLine->product_id)) {
                if ($salesOrderLine->product->weight <= $maxWeight) {
                    foreach (range(1, $salesOrderLine->quantity) as $value) {
                        if (($orders[$index]['total_weight'] + $salesOrderLine->product->weight) <= $maxWeight) {
                            $orders[$index]['total_weight'] += $salesOrderLine->product->weight;

                            if (! isset($orders[$index]['lines'][$salesOrderLine->id])) {
                                $orders[$index]['lines'][$salesOrderLine->id] = [
                                    'id' => $salesOrderLine->id,
                                    'quantity' => 1,
                                    'sku' => $salesOrderLine->product->sku,
                                    'description' => $salesOrderLine->description,
                                ];
                            } else {
                                $orders[$index]['lines'][$salesOrderLine->id]['quantity'] += 1;
                            }
                        } else {
                            $orders[++$index] = [
                                'total_weight' => $salesOrderLine->product->weight,
                                'lines' => [
                                    $salesOrderLine->id => [
                                        'id' => $salesOrderLine->id,
                                        'quantity' => 1,
                                        'sku' => $salesOrderLine->product->sku,
                                        'description' => $salesOrderLine->description,
                                    ],
                                ],
                            ];
                        }
                    }
                }
            }
        }

        return response()->json($orders);
    }

    /**
     * Splitting an order by default supplier.
     */
    public function splitByDefaultSupplier($salesOrderId): JsonResponse
    {
        $salesOrder = SalesOrder::with([
            'salesOrderLines',
            'salesOrderLines.product',
            'salesOrderLines.product.defaultSupplierProduct',
        ])->findOrFail($salesOrderId);

        $suppliers = Supplier::all();

        $orders = [];
        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            // check the sales order line is a product
            if ($salesOrderLine->nominal_code_id == $this->productLineItemNominalCode && ! empty($salesOrderLine->product_id)) {
                // check the product has a default supplier product
                if ($salesOrderLine->product->defaultSupplierProduct) {
                    if (isset($orders[$salesOrderLine->product->defaultSupplierProduct->supplier_id])) {
                        $orders[$salesOrderLine->product->defaultSupplierProduct->supplier_id]['total_weight'] += $salesOrderLine->product->weight * $salesOrderLine->quantity;
                        $orders[$salesOrderLine->product->defaultSupplierProduct->supplier_id]['lines'][$salesOrderLine->id] = [
                            'id' => $salesOrderLine->id,
                            'quantity' => $salesOrderLine->quantity,
                            'sku' => $salesOrderLine->product->sku,
                            'description' => $salesOrderLine->description,
                        ];
                    } else {
                        $orders[$salesOrderLine->product->defaultSupplierProduct->supplier_id] = [
                            'total_weight' => $salesOrderLine->product->weight * $salesOrderLine->quantity,
                            'lines' => [
                                $salesOrderLine->id => [
                                    'id' => $salesOrderLine->id,
                                    'quantity' => $salesOrderLine->quantity,
                                    'sku' => $salesOrderLine->product->sku,
                                    'description' => $salesOrderLine->description,
                                ],
                            ],
                            'supplier' => [
                                'id' => $salesOrderLine->product->defaultSupplierProduct->supplier_id,
                                'name' => $suppliers->firstWhere('id', $salesOrderLine->product->defaultSupplierProduct->supplier_id)->name,
                            ],
                        ];
                    }
                }
            }
        }

        return response()->json($orders);
    }

    /**
     * Splitting an order by one(line) in each order.
     */
    public function splitOneInEachOrder($salesOrderId): JsonResponse
    {
        $salesOrder = SalesOrder::with(['salesOrderLines', 'salesOrderLines.product'])->findOrFail($salesOrderId);

        $index = 1;
        $orders = [];

        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            if ($salesOrderLine->nominal_code_id == $this->productLineItemNominalCode && ! empty($salesOrderLine->product_id)) {

                foreach (range(1, $salesOrderLine->quantity) as $value) {
                    $orders[$index++] = [
                        'total_weight' => $salesOrderLine->product->weight,
                        'lines' => [
                            $salesOrderLine->id => [
                                'id' => $salesOrderLine->id,
                                'quantity' => 1,
                                'sku' => $salesOrderLine->product->sku,
                                'description' => $salesOrderLine->description,
                            ],
                        ],
                    ];
                }

            }
        }

        return response()->json($orders);
    }

    /**
     * Get the coordinates from an address.
     *
     *
     * @throws Exception
     */
    private function getCoordinatesFromAddress(?Address $address): ?array
    {
        if (empty(env('GOOGLE_GEOCODE_KEY'))) {
            throw new Exception(__('messages.missing_geocode_key'));
        }

        if (is_null($address)) {
            return null;
        }

        $curl = curl_init();

        $address1 = urlencode($address->address1);
        $address2 = urlencode($address->zip.' '.$address->city);

        $url = 'https://maps.googleapis.com/maps/api/geocode/json?';
        $url .= "address=$address1,$address2,$address->country_code&";
        $url .= "components=country:$address->country_code&";
        $url .= 'key='.env('GOOGLE_GEOCODE_KEY');

        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        $response = curl_exec($curl);
        curl_close($curl);

        $response = json_decode($response, true);
        if ($response['status'] == 'OK') {
            return $response['results'][0]['geometry']['location'];
        }

        return null;
    }
}
