<?php

namespace App\Http\Controllers\Traits;

use App\Models\Concerns\Archive;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

/**
 * Trait BulkOperation.
 */
trait BulkOperation
{
    protected $BULK_ARCHIVE = 'archive';

    protected $BULK_UN_ARCHIVE = 'unarchived';

    protected $BULK_DELETE = 'delete';

    public function bulkOperation(Request $request, string $bulkProcess)
    {
        // validate
        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
            'ids.*' => 'integer|exists:'.( new $this->model_path )->getTable().',id',
        ]);

        if ($request->has('ids')) {
            $ids = array_unique($request->input('ids', []));
            $data['totalCount'] = count($ids);

            switch ($bulkProcess) {
                case $this->BULK_ARCHIVE:
                    $data['failCount'] = $this->archiveByIds($ids, $this->model_path);
                    break;
                case $this->BULK_UN_ARCHIVE:
                    $data['failCount'] = $this->unArchiveByIds($ids, $this->model_path);
                    break;
                case $this->BULK_DELETE:
                    $data['failCount'] = $this->deleteByIds($ids, $this->model_path);
                    break;
            }
        } else {
            switch ($bulkProcess) {
                case $this->BULK_ARCHIVE:
                    $data = $this->archiveByFilters($request, $this->model_path);
                    break;
                case $this->BULK_UN_ARCHIVE:
                    $data = $this->unArchiveByFilters($request, $this->model_path);
                    break;
                case $this->BULK_DELETE:
                    $data = $this->deleteByFilters($request, $this->model_path);
                    break;
            }
        }

        $resource = str_replace('_', ' ', ( new $this->model_path )->getTable());

        // all models are already archived
        if ($data['failCount'] == $data['totalCount']) {
            switch ($bulkProcess) {
                case $this->BULK_ARCHIVE:
                    return $this->response->setMessage(__('messages.failed.bulk_archive', ['resource' => $resource]));
                case $this->BULK_UN_ARCHIVE:
                    return $this->response->setMessage(__('messages.failed.bulk_un_archive', ['resource' => $resource]));
                case $this->BULK_DELETE:
                    return $this->response->setMessage(__('messages.failed.bulk_delete', ['resource' => $resource]));
            }
        }

        switch ($bulkProcess) {
            case $this->BULK_ARCHIVE:
                return $this->response->setMessage(__('messages.success.bulk_archive', [
                    'total_count' => $data['totalCount'],
                    'success_count' => ($data['totalCount'] - $data['failCount']),
                    'resource' => $resource,
                ]));
            case $this->BULK_UN_ARCHIVE:
                return $this->response->setMessage(__('messages.success.bulk_un_archive', [
                    'total_count' => $data['totalCount'],
                    'success_count' => ($data['totalCount'] - $data['failCount']),
                    'resource' => $resource,
                ]));
            case $this->BULK_DELETE:
                return $this->response->setMessage(__('messages.success.bulk_delete', [
                    'total_count' => $data['totalCount'],
                    'success_count' => ($data['totalCount'] - $data['failCount']),
                    'resource' => $resource,
                ]));
        }
    }

    /**
     * archive resource by ids.
     */
    private function archiveByIds(array $ids, string $model): int
    {
        $failCount = 0;

        $instances = $model::with([])->whereIn('id', $ids)->get();

        foreach ($instances as $instance) {
            $failCount += $this->archiveInstance($instance) ? 0 : 1;
        }

        return $failCount;
    }

    /**
     * unarchived resource by ids.
     */
    private function unArchiveByIds(array $ids, string $model): int
    {
        $failCount = 0;

        $instances = $model::with([])
            ->whereIn('id', $ids)

      // Next line commented out because all fields are needed in case we have
      // to restore the model's fulltext index record in SphynxSearch:

      //->select('id', 'archived_at')
            ->get();

        foreach ($instances as $instance) {
            $failCount += $this->unArchiveInstance($instance) ? 0 : 1;
        }

        return $failCount;
    }

    /**
     * archived resource by request.
     */
    private function archiveByFilters($request, string $model): array
    {
        $data = ['totalCount' => 0, 'failCount' => 0, 'failedIds' => []];

        do {
            $instances = $model::with([])
                ->filter($request)
                ->archived(0)
                ->limit(30)->select('id', 'archived_at')
                ->whereNotIn('id', $data['failedIds'])
                ->get();

            foreach ($instances as $instance) {
                if (! $this->archiveInstance($instance)) {
                    $data['failCount'] += 1;
                    $data['failedIds'][] = $instance->id;
                }
            }

            $data['totalCount'] += $instances->count();
        } while ((bool) $instances->count());

        return $data;
    }

    /**
     * unarchived resource by request.
     */
    private function unArchiveByFilters($request, string $model): array
    {
        $data = ['totalCount' => 0, 'failCount' => 0, 'failedIds' => []];

        do {
            $instances = $model::with([])
                ->filter($request)
                ->archived(1)
                ->limit(30)

        // Next line commented out because all fields are needed
        // in case we have to restore the model's fulltext index
        // record in SphynxSearch:

        //->select('id', 'archived_at')
                ->whereNotIn('id', $data['failedIds'])
                ->get();

            foreach ($instances as $instance) {
                if (! $this->unArchiveInstance($instance)) {
                    $data['failCount'] += 1;
                    $data['failedIds'][] = $instance->id;
                }
            }

            $data['totalCount'] += $instances->count();
        } while ((bool) $instances->count());

        return $data;
    }

    private function deleteByIds(array $ids, string $model)
    {
        $failCount = 0;

        $instances = $model::with([])->whereIn('id', $ids)->get();

        foreach ($instances as $instance) {
            $failCount += $this->deleteInstance($instance) ? 0 : 1;
        }

        return $failCount;
    }

    private function deleteByFilters($request, string $model)
    {
        $data = ['totalCount' => 0, 'failCount' => 0, 'failedIds' => []];

        $usedTraits = class_uses_recursive($model);
        do {
            $instances = $model::with([])
                ->filter($request)->addRelations($request)->addCustomColumns($request)
                ->when(in_array(Archive::class, $usedTraits) && $request->has('archived'), function ($query) use ($request) {
                    $query->archived($request->get('archived', 0));
                })
                ->limit(30)
                ->whereNotIn('id', $data['failedIds'])
                ->get();

            foreach ($instances as $instance) {
                if (! $this->deleteInstance($instance)) {
                    $data['failCount'] += 1;
                    $data['failedIds'][] = $instance->id;
                }
            }

            $data['totalCount'] += $instances->count();
        } while ((bool) $instances->count());

        return $data;
    }

    /**
     * validation archiving process from model.
     */
    protected function archiveInstance(Model $instance): bool
    {
        $resource = Str::singular($instance->getTable());

        $response = $instance->archive();

        if (is_string($response)) {
            $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($response, ucfirst(Str::camel($resource)).Response::CODE_UNACCEPTABLE, "ids.{$instance->id}.id");

            return false;
        }

        if (! $response) {
            // add warning
            $this->response->addWarning(__('messages.failed.already_archive', [
                'resource' => str_replace('_', ' ', $resource),
                'id' => $instance->id,
            ]), ucfirst(Str::camel($resource)).Response::CODE_ALREADY_ARCHIVED, "ids.{$instance->id}.id");

            return false;
        }

        return true;
    }

    /**
     * validation un archiving process from model.
     */
    private function unArchiveInstance(Model $instance): bool
    {
        $resource = Str::singular($instance->getTable());

        if (! $instance->unarchived()) {
            // add warning
            $this->response->addWarning(__('messages.failed.unarchived', [
                'resource' => str_replace('_', ' ', $resource),
                'id' => $instance->id,
            ]), ucfirst(Str::camel($resource)).Response::CODE_ALREADY_UNARCHIVED, "ids.{$instance->id}.id");

            return false;
        }

        return true;
    }

    /**
     * @throws Exception
     */
    protected function deleteInstance(Model $instance): bool
    {
        $instance->refresh(); // Refresh to make sure checks on properties can be performed.
        $reasons = $instance->delete();

        // check if the resource is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addWarning($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, "ids.{$instance->id}.{$key}", [
                    'resource' => $this->getModelName($this->model_path),
                    'resource_id' => $instance->id,
                ]);
            }

            return false;
        }

        return true;
    }

    /**
     * convert camel case to snake case and make it plural string.
     */
    private function tableName($input): string
    {
        preg_match_all('!([A-Z][A-Z0-9]*(?=$|[A-Z][a-z0-9])|[A-Za-z][a-z0-9]+)!', $input, $matches);
        $ret = $matches[0];
        foreach ($ret as &$match) {
            $match = $match == strtoupper($match) ? strtolower($match) : lcfirst($match);
        }

        return Str::plural(implode('_', $ret));
    }

    private function getModelName(string $model_path)
    {
        $arr = explode('\\', $model_path);

        return $arr[count($arr) - 1];
    }
}
