<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\TaxRate\StoreTaxRateRequest;
use App\Http\Resources\TaxRateResource;
use App\Models\SalesOrder;
use App\Models\TaxRate;
use App\Response;
use Illuminate\Http\Request;

class TaxRateController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = TaxRate::class;

    protected $resource = 'tax rate';

    protected function getModel(): string
    {
        return TaxRate::class;
    }

    protected function getResource(): string
    {
        return TaxRateResource::class;
    }

    /**
     * Get a Tax Rate by ID.
     *
     * @param  TaxRate  $taxRate
     */
    public function show($id): Response
    {
        /**
         * Get Tax Rate by id with relations Or fail 404 not found.
         */
        $taxRate = TaxRate::with(['taxRateLines'])->addRelations()->findOrFail($id);

        return $this->response->addData(TaxRateResource::make($taxRate));
    }

    /**
     * Create a new tag.
     */
    public function store(StoreTaxRateRequest $request): Response
    {
        $taxLinesRequest = @$request->validated()['tax_rate_lines'];
        $taxRate = TaxRate::create($request->validated());

        if ($taxLinesRequest) {
            foreach ($taxLinesRequest as $taxLine) {
                $taxRate->taxRateLines()->create($taxLine);
            }
        }

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => 'tax rate']))
            ->addData(TaxRateResource::make($taxRate->load('taxRateLines')));
    }

    public function updateAccountingMapping(Request $request)
    {
        $request->validate([
            'tax_rates' => 'required|array',
            'tax_rates.*.id' => 'required|exists:tax_rates,id',
            'tax_rates.*.accounting_integration_id' => 'nullable',
            'tax_rates.*.accounting_integration_type' => 'nullable',
        ]);

        foreach ($request->get('tax_rates') as $taxRateIteration) {
            $paymentType = TaxRate::findOrFail($taxRateIteration['id']);
            $paymentType->accounting_integration_id = @$taxRateIteration['accounting_integration_id'];
            $paymentType->accounting_integration_type = @$taxRateIteration['accounting_integration_type'];
            $paymentType->update();
        }

        return $this->response->success();
    }

    public function destroy($taxRateId)
    {
        $taxRate = TaxRate::with([])->findOrFail($taxRateId);

        $salesOrderExist = SalesOrder::whereHas('taxRate', function ($query) use ($taxRate) {
            $query->whereId($taxRate->id);
        })
            ->exists();

        if ($salesOrderExist) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.tax-rate.delete_failed', ['id' => $taxRate->id]));
        } else {
            $taxRate->delete();

            return $this->response->setMessage(__('messages.tax-rate.delete_success', ['id' => $taxRate->id]));
        }
    }

    public function archive($taxRateId)
    {
        $taxRate = TaxRate::with([])->findOrFail($taxRateId);

        if ($taxRate->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => 'tax-rate',
                'id' => $taxRate->id,
            ]));
        }

        return $this->response->addWarning(__('messages.failed.already_archive', [
            'resource' => 'tax-rate',
            'id' => $taxRate->id,
        ]), 'TaxRate'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $taxRateId]);
    }

    public function unarchived($taxRateId)
    {
        $taxRate = TaxRate::findOrFail($taxRateId);

        if ($taxRate->unarchived()) {
            return $this->response->setMessage(__('messages.success.unarchived', [
                'resource' => $this->resource,
                'id' => $taxRate->id,
            ]));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $taxRate->id,
            ]), 'TaxRate'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $taxRateId]);
    }

    public function bulkArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    public function bulkDestroy(Request $request)
    {
        $salesOrderExist = SalesOrder::whereHas('taxRate', function ($query) use ($request) {
            $query->whereIn('id', $request->ids);
        })
            ->exists();

        if ($salesOrderExist) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.tax-rate.delete_failed', ['id' => '']));
        } else {
            return $this->bulkOperation($request, $this->BULK_DELETE);
        }
    }

    public function bulkUnarchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }
}
