<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Resources\TagResource;
use App\Models\Tag;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class TagController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = Tag::class;

    private $resource = 'tag';

    /**
     * Display tag.
     */
    public function show($tagId): Response
    {
        $tag = Tag::with([])->withCount(['products', 'salesOrders', 'purchaseOrders'])->findOrFail($tagId);

        return $this->response->addData(TagResource::make($tag));
    }

    /**
     * Create a new tag.
     */
    public function store(Request $request): Response
    {
        $request->validate(['name' => 'required|unique:tags,name']);

        $tag = Tag::create($request->only(['name']));

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->response]))
            ->addData(TagResource::make($tag));
    }

    /**
     * Update the tag.
     */
    public function update(Request $request, $tagId): Response
    {
        $request->validate(['name' => 'required|unique:tags,name,'.$tagId]);

        $tag = Tag::with([])->findOrFail($tagId);
        $tag->fill($request->only(['name']));
        $tag->save();

        return $this->response->setMessage(__('messages.success.update', [
            'resource' => $this->response,
            'id' => $tag->name,
        ]))
            ->addData(TagResource::make($tag));
    }

    /**
     * Delete tag.
     *
     *
     * @throws Exception
     */
    public function destroy(Tag $tag): Response
    {
        $reasons = $tag->delete();

        // check if the tag is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['tag_id' => $tag->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'tag',
                    'id' => $tag->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'tag',
            'id' => $tag->name,
        ]));
    }

    /**
     * Bulk delete tags.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Archive the tag.
     */
    public function archive(Tag $tag): Response
    {
        if ($tag->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $tag->name,
                ]))
                ->addData(TagResource::make($tag));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $tag->name,
            ]), 'Tag'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $tag->id])
            ->addData(TagResource::make($tag));
    }

    /**
     * Unarchived the tag.
     */
    public function unarchived(Tag $tag): Response
    {
        if ($tag->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $tag->name,
                ]))
                ->addData(TagResource::make($tag));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $tag->name,
            ]), 'Tag'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $tag->id])
            ->addData(TagResource::make($tag));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:tags,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $tags = Tag::with([])->whereIn('id', $ids)->get();
        foreach ($tags as $key => $tag) {
            $isUsed = $tag->isUsed();

            $result[$key] = $tag->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Tag::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return TagResource::class;
    }
}
