<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Requests\StoreWarehouse;
use App\Http\Resources\SupplierWarehouseResource;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\SupplierWarehouse;
use App\Models\Warehouse;
use App\Response;
use Illuminate\Http\Request;

/**
 * Class SupplierWarehouseController.
 */
class SupplierWarehouseController extends WarehouseController
{
    use DataTable;

    /**
     * @return $this|Response|mixed
     */
    public function supplierWarehouses(Request $request, $supplierId)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications(SupplierWarehouse::class));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request, SupplierWarehouse::class);

        $warehouse = SupplierWarehouse::with(['address', 'supplier'])->where('supplier_id', e($supplierId));

        $warehouse->filter($request);
        $warehouse->addRelations($request);
        $warehouse->addCustomColumns();
        $warehouse->sort($request);

        // paginate with limit per page (default 10)
        $warehouse = DataTableConfiguration::paginate($warehouse);

        return SupplierWarehouseResource::collectionWithTableSpecifications($warehouse, SupplierWarehouse::class);
    }

    /**
     * Creates a new warehouse for a supplier.
     * If it's the first warehouse for the supplier,
     * it makes it the default warehouse.
     */
    public function storeWarehouse(StoreWarehouse $request, Supplier $supplier): Response
    {
        $inputs = $request->validated();

        // Indicator to check if this is the supplier's first warehouse or not.
        $supplierHasWarehouses = $supplier->hasWarehouses();

        // Create the supplier warehouse in the repository
        $warehouse = $this->warehouses->createWarehouse(array_merge($inputs, [
            'supplier_id' => e($supplier->id),
            'type' => Warehouse::TYPE_SUPPLIER,
        ]));

        // Initialize supplier inventory for the new supplier warehouse
        $this->inventories->initializeInventoryForSupplierWarehouse($warehouse);

        // We update the supplier's default warehouse if the created warehouse is default
        // or if this is the supplier's first warehouse
        if ((isset($inputs['is_default']) && $inputs['is_default']) || ! $supplierHasWarehouses) {
            $supplier->setDefaultWarehouse($warehouse->id);
        }

        // Load the warehouse address
        $warehouse->load('address');

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(SupplierWarehouseResource::make($warehouse));
    }

    public function linkedOrders(Warehouse $warehouse): Response
    {
        $linkedOrders = $warehouse->outgoingPurchaseOrders()->where('order_status', '!=', PurchaseOrder::STATUS_CLOSED)->count();

        return $this->response->addData(['linked_orders' => $linkedOrders]);
    }

    /**
     * @return string
     */
    public function getResource()
    {
        return SupplierWarehouseResource::class;
    }
}
