<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreSupplierProduct;
use App\Http\Resources\SupplierProductResource;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\SupplierProduct;
use App\Repositories\SupplierInventoryRepository;
use App\Response;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class SupplierProductController extends Controller
{
    /**
     * @var SupplierInventoryRepository
     */
    private $inventories;

    /**
     * SupplierProductController constructor.
     */
    public function __construct(SupplierInventoryRepository $inventories)
    {
        $this->inventories = $inventories;
        parent::__construct();
    }

    /**
     * Retrieve supplier's products.
     */
    public function index(Request $request, $supplierId): AnonymousResourceCollection
    {
        $supplierProducts = SupplierProduct::with([
            'product.suppliersInventory',
            'supplierPricingTiers',
        ])
            ->whereHas('product', function (Builder $query) {
                $query->where('type', 'standard');
            })
            ->where('supplier_id', $supplierId);

        if ($query = $request->query('query')) {
            $supplierProducts->where(function (Builder $builder) use ($query) {
                $builder->where('supplier_sku', 'like', $query.'%');
                $builder->orWhere(function (Builder $builder) use ($query) {
                    /**/
                    // And an 'or' filter for product name and product sku
                    $sku_products = env('SPHINX_INDEX_PREFIX', 'sku_').'products';
                    if (SphinxSearch::indexEnabled($sku_products)) {
                        SphinxSearch::filter(
                            $builder,
                            $sku_products,
                            [
                                'query' => $query,
                                'fk_field' => 'product_id',
                            ]
                        );

                    // Same as:
                    //
                    // $supplierProducts->whereIn('product_id', function($q) use ($query) {
                    //     $q->select(\DB::raw(SphinxSearch::getRawSelect('sku_products')))
                    //         ->where('query', SphinxSearch::getRawQueryString(['query'=>$query]));
                    // });
                    } else {
                        $builder->whereHas('product', function (Builder $builder) use ($query) {
                            return $builder->where('name', 'like', $query.'%')
                                ->orWhere('sku', 'like', $query.'%')
                                ->orWhere('barcode', 'like', $query.'%');
                        });
                    }
                });
            });
        }
        /**/
        /*
              $supplierProducts->join('products AS product_filter', function($join) use ($query) {
                  if (SphinxSearch::indexEnabled(## 'sku_products')) {
                    SphinxSearch::join('sku_products',$join, ['fk_field'=>'product_filter.id','query'=>$query]);
                  } else {
                    $join->where('product_filter.name', 'like', $query . '%')->orWhere('product_filter.sku', 'like', $query . '%');
                  }
              });
         */

        $limit = $request->query('limit') ?? null;
        if ($limit && $limit < 0) {
            $data = $supplierProducts->get();
        } else {
            $data = $supplierProducts->paginate($limit);
        }

        return SupplierProductResource::collection($data)->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Store a new supplier product.
     */
    public function store(StoreSupplierProduct $request): Response
    {
        $supplierProduct = new SupplierProduct($request->validated());
        $supplierProduct->save();

        // Initialize inventory for the supplier product
        $this->inventories->initializeInventoryForSupplierProduct($supplierProduct);

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => 'supplier product']))
            ->addData(SupplierProductResource::make($supplierProduct));
    }

    /**
     * Update supplier product.
     */
    public function update(StoreSupplierProduct $request, $supplierProductId): Response
    {
        $supplierProduct = SupplierProduct::with([])->findOrFail($supplierProductId);

        // If the product or supplier is changing, we need to handle supplier inventories
        // for the updated supplier product
        if ($this->changingSupplierOrProduct($supplierProduct, $request->validated())) {
            $this->handleSupplierInventoryChanges($supplierProduct, $request->validated());

            // Add a warning about the changes
            $this->response->addWarning(__('messages.supplier_product.reset_inventory', [
                'supplier_product_id' => $supplierProduct->id,
            ]), 'Warehouse'.Response::CODE_SUPPLIER_INVENTORY_RESET, 'id', ['id' => $supplierProduct->id]);
        }

        // fill data and save
        $supplierProduct->fill($request->all());
        $supplierProduct->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => 'supplier product',
                'id' => $supplierProduct->supplier_sku ?: $supplierProductId,
            ]))
            ->addData(SupplierProductResource::make($supplierProduct));
    }

    private function changingSupplierOrProduct(SupplierProduct $supplierProduct, array $payload): bool
    {
        return (isset($payload['supplier_id']) && $payload['supplier_id'] !== $supplierProduct->supplier_id) ||
           (isset($payload['product_id']) && $payload['product_id'] !== $supplierProduct->product_id);
    }

    private function handleSupplierInventoryChanges(SupplierProduct $supplierProduct, array $payload)
    {
        // Since either the product or supplier is changing, we clear the supplier inventories
        // of the supplier product and re-initialize them.
        $this->inventories->clearInventoryForSupplierProduct($supplierProduct->supplier_id, $supplierProduct->product_id);

        // Re-initialize supplier inventory
        $supplierProduct->fill($payload);
        $this->inventories->initializeInventoryForSupplierProduct($supplierProduct);
    }
}
