<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreSupplierPricing;
use App\Http\Resources\SupplierPricingTierResource;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class SupplierPricingController extends Controller
{
    use bulkOperation
  {
      archiveInstance as originalArchiveInstance;
  }
    use DataTable, ImportsData;

    protected $model_path = SupplierPricingTier::class;

    private $resource = 'supplier pricing tier';

    /**
     * Create new supplier product pricing tiers.
     */
    public function store(StoreSupplierPricing $request): Response
    {
        $productPricingTier = new SupplierPricingTier($request->all());
        $productPricingTier->currency_code = $request->input('currency_code');
        $productPricingTier->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(SupplierPricingTierResource::make($productPricingTier));
    }

    public function update(StoreSupplierPricing $request, $supplierId)
    {
        $supplierPricingTier = SupplierPricingTier::with([])->findOrFail($supplierId);
        $supplierPricingTier->name = $request->input('name');
        $supplierPricingTier->currency_code = $request->input('currency_code');
        $supplierPricingTier->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $supplierPricingTier->name,
            ]))
            ->addData(SupplierPricingTierResource::make($supplierPricingTier));
    }

    public function fetchForSupplier(Request $request, $supplierId)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications(SupplierPricingTier::class));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request, SupplierPricingTier::class);

        $supplier = Supplier::with([])->findOrFail(e($supplierId));

        $tiers = $supplier->pricingTiers();

        $tiers->filter($request);
        $tiers->addRelations($request);
        $tiers->addCustomColumns();
        $tiers->sort($request);

        return SupplierPricingTierResource::collectionWithTableSpecifications($tiers->paginate(10), SupplierPricingTier::class);
    }

    public function archive(SupplierPricingTier $supplierPricingTier)
    {
        $this->response->addData(SupplierPricingTierResource::make($supplierPricingTier));

        // prevent archive the default pricing tier
        if ($supplierPricingTier->is_default) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.supplier.archive_pricing_failed', ['id' => $supplierPricingTier->name]), 'SupplierPricingTier'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $supplierPricingTier->id]);
        }

        if ($supplierPricingTier->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $supplierPricingTier->name,
                ]));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $supplierPricingTier->name,
            ]), 'SupplierPricingTier'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $supplierPricingTier->id]);
    }

    public function unarchived(SupplierPricingTier $supplierPricingTier)
    {
        if ($supplierPricingTier->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $supplierPricingTier->name,
                ]))
                ->addData(SupplierPricingTierResource::make($supplierPricingTier));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $supplierPricingTier->name,
            ]), 'SupplierPricingTier'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $supplierPricingTier->id])
            ->addData(SupplierPricingTierResource::make($supplierPricingTier));
    }

    /**
     * Set supplier pricing tier as default tier.
     */
    public function setAsDefault(SupplierPricingTier $supplierPricingTier): Response
    {
        // check is archived
        if ($supplierPricingTier->isArchived()) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.supplier.set_default_pricing_failed', ['id' => $supplierPricingTier->name]), 'SupplierPricingTier'.Response::CODE_UNACCEPTABLE, 'id', ['id' => $supplierPricingTier->id])
                ->addData(SupplierPricingTierResource::make($supplierPricingTier));
        }

        // set default to true
        $supplierPricingTier->is_default = true;
        $supplierPricingTier->save();

        return $this->response
            ->setMessage(__('messages.success.default', [
                'resource' => $this->resource,
                'id' => $supplierPricingTier->name,
            ]))
            ->addData(SupplierPricingTierResource::make($supplierPricingTier));
    }

    /**
     * Delete Supplier Pricing Tier.
     *
     *
     * @throws Exception
     */
    public function destroy(SupplierPricingTier $supplierPricingTier): Response
    {
        $reasons = $supplierPricingTier->delete();

        // check if the resource is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['supplier_pricing_tier_id' => $supplierPricingTier->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $supplierPricingTier->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $supplierPricingTier->name,
        ]));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete resources.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:supplier_pricing_tiers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $supplierPricingTiers = SupplierPricingTier::with([])->whereIn('id', $ids)->select('id', 'name', 'is_default')->get();
        foreach ($supplierPricingTiers as $key => $supplierPricingTier) {
            $isUsed = $supplierPricingTier->isUsed();

            $result[$key] = $supplierPricingTier->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * check the possibility of archiving.
     */
    public function isArchivable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:supplier_pricing_tiers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $supplierPricingTiers = SupplierPricingTier::with([])->whereIn('id', $ids)->select('id', 'name', 'is_default')->get();
        foreach ($supplierPricingTiers as $key => $supplierPricingTier) {
            $result[$key] = $supplierPricingTier->only('id', 'name', 'is_default');
            $result[$key]['archivable'] = ! $supplierPricingTier->is_default;
            $result[$key]['reason'] = $supplierPricingTier->is_default ? ['is_default' => __('messages.supplier.archive_pricing_failed', ['id' => $supplierPricingTier->name])] : null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function archiveInstance(Model $instance)
    {
        // prevent archive the default pricing tier
        if ($instance->is_default) {
            $this->response->addWarning(__('messages.supplier.archive_pricing_failed', ['id' => $instance->name]), 'SupplierPricingTier'.Response::CODE_UNACCEPTABLE, "ids.{$instance->id}.id", ['id' => $instance->id]);

            return false;
        }

        return $this->originalArchiveInstance($instance);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return SupplierPricingTier::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return SupplierPricingTierResource::class;
    }
}
