<?php

namespace App\Http\Controllers\Shopify;

use App\Helpers;
use App\Http\Controllers\Controller;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\Shopify\ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Repositories\Shopify\ShopifyBulkOperationRepository;
use App\Repositories\Shopify\ShopifyWebhookEventRepository;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Shopify\Jobs\ShopifyOrderWebhookEventJob;

class ShopifyWebhookController extends Controller
{
    public function __construct(protected ShopifyWebhookEventRepository $webhookEventRepository)
    {
        parent::__construct();
    }

    public function inventoryItemDelete(Request $request, $integrationInstanceId): JsonResponse
    {
        /** @var ShopifyProduct $product */
        $product = ShopifyProduct::with([])->where('integration_instance_id', (int) $integrationInstanceId)
            ->where('inventory_item_id', $request->input('id'))
            ->first();
        if ($product) {
            if ($product->update(['removed_from_shopify' => 1])) {
                $product->productListing()->delete();
            }
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * @throws BindingResolutionException
     */
    public function bulkOperationsFinish(Request $request, $integrationInstanceId): JsonResponse
    {
        $inputs = $request->all();

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::with('integration')->findOrFail($integrationInstanceId);

        app()->make(ShopifyBulkOperationRepository::class)->update($integrationInstance, $inputs);

        return response()->json(['status' => 'success']);
    }

    public function productCreate(Request $request, $integrationInstanceId): JsonResponse
    {
        return response()->json(['status' => 'success']);
    }

    public function productUpdate(Request $request, $integrationInstanceId): JsonResponse
    {
        /*
         * Kalvin:
         * Until Shopify updates product webhooks/updated_at to not update for inventory updates, we are disabling
         * this webhook and its handling
         * See ref: https://community.shopify.com/c/shopify-apis-and-sdks/will-products-update-webhook-always-fire-when-an-inventory-level/td-p/572280
         */
        /*$integrationInstance = IntegrationInstance::with('integration')->findOrFail($integrationInstanceId);

        GetProducts::createProductsByVariations($integrationInstance, $request->toArray());
        */
        return response()->json(['status' => 'success']);
    }

    public function inventoryLevelUpdate(Request $request, $integrationInstanceId): JsonResponse
    {
        $documentId = ShopifyProduct::with([])->where('integration_instance_id', (int) $integrationInstanceId)
            ->where('inventory_item_id', $request->input('inventory_item_id'))
            ->value('_id');

        ProductListing::with([])->where('document_id', $documentId)
            ->update([
                'sales_channel_qty' => $request->input('available'),
                'sales_channel_qty_last_updated' => Carbon::parse($request->input('updated_at'))->timezone('UTC'),
            ]);

        return response()->json(['status' => 'success']);
    }

    public function orderCreate(Request $request, $integrationInstanceId): JsonResponse
    {
        $order = $request->except('token');

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::query()->findOrFail($integrationInstanceId);

        ShopifyOrderWebhookEventJob::dispatch($integrationInstance, ShopifyWebhook::TOPIC_ORDERS_CREATE, $order);

        return response()->json(['status' => 'success']);
    }

    public function orderUpdated(Request $request, $integrationInstanceId): JsonResponse
    {
        $order = $request->except('token');

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::query()->findOrFail($integrationInstanceId);

        if (! isset($order['name'])) {
            Helpers::slack('Shopify orderUpdated webhook hit with unexpected payload: '.$request->getContent());
        }

        ShopifyOrderWebhookEventJob::dispatch($integrationInstance, ShopifyWebhook::TOPIC_ORDERS_UPDATED, $order);

        return response()->json(['status' => 'success']);
    }

    public function orderTransactionCreate(Request $request, $integrationInstanceId): JsonResponse
    {
        $transaction = $request->except('token');

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::query()->findOrFail($integrationInstanceId);

        ShopifyOrderWebhookEventJob::dispatch($integrationInstance, ShopifyWebhook::TOPIC_ORDER_TRANSACTIONS_CREATE, $transaction);

        return response()->json(['status' => 'success']);
    }
}
