<?php

namespace App\Http\Controllers\Shopify;

use App\Http\Controllers\IntegrationInstanceController;
use App\Jobs\Shopify\ShopifyGetLocationsJob;
use App\Jobs\Shopify\ShopifyGetShippingServices;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyLocation;
use App\Response;
use Illuminate\Http\Request;

class ShopifyIntegrationController extends IntegrationInstanceController
{
    public function downloadShippingServices(Request $request, IntegrationInstance $integrationInstance): Response
    {
        if ($integrationInstance->integration->name !== Integration::NAME_SHOPIFY) {
            return $this->response->addError(
                'This is only available for Shopify Integrations.',
                Response::CODE_UNACCEPTABLE,
                'DownloadShippingServices'
            );
        }

        dispatch_sync((new ShopifyGetShippingServices($integrationInstance))->onConnection('sync'));

        return $this->shippingServiceFromSalesChannel($request, $integrationInstance);
    }

    public function downloadLocations(IntegrationInstance $integrationInstance)
    {
        ShopifyGetLocationsJob::dispatch($integrationInstance)->onConnection('sync');
        // refresh from DB
        $integrationInstance->refresh();

        return $this->response->addData([
            'shopifyLocations' => ShopifyLocation::with([])->where('integration_instance_id', $integrationInstance->id)->get()->toArray(),
            'inventoryLocations' => $integrationInstance->integration_settings['inventory']['locations'] ?? [],
        ]);
    }

    public function getLocations(IntegrationInstance $integrationInstance)
    {
        return $this->response->addData([
            'shopifyLocations' => ShopifyLocation::with([])->where('integration_instance_id', $integrationInstance->id)->get()->toArray(),
            'inventoryLocations' => $integrationInstance->integration_settings['inventory']['locations'] ?? [],
        ]);
    }
}
