<?php

namespace App\Http\Controllers\ShipStation;

use App\Exceptions\ApiException;
use App\Http\Controllers\Controller;
use App\Http\Requests\MapShipstationWarehouseRequest;
use App\Http\Resources\ShipstationWarehouseMappingResource;
use App\Http\Resources\ShipstationWarehouseResource;
use App\Models\IntegrationInstance;
use App\Models\ShipStation\ShipstationWarehouse;
use App\Repositories\Shipstation\ShipstationRepository;
use App\Response;
use App\Services\ShippingProvider\ShipStationManager;
use Illuminate\Http\Request;

class WarehouseController extends Controller
{

    public function __construct(
        private readonly ShipStationManager $manager,
        private readonly ShipstationRepository $repository,
    )
    {
        parent::__construct();
    }

    public function index(){
        return ShipstationWarehouseResource::collection(
            $this->repository->getWarehouses()
        );
    }


    public function getWarehouseMappings(): ?Response
    {
        return $this->response->addData(
                ShipstationWarehouseMappingResource::collection(
                    $this->repository->getWarehouseMappings()
                )
            );
    }


    public function mapWarehouses(MapShipstationWarehouseRequest $request){
        try{
            $this->repository->createWarehouseMappings($request->validated('mapping'));
            return $this->response
                ->addData(
                    ShipstationWarehouseMappingResource::collection(
                        $this->repository->getWarehouseMappings()
                    )
                )
                ->success(message: 'Shipstation warehouses mapped successfully.');
        }catch (\Exception $e){
            return $this->response->addError(
                "There was a problem mapping Shipstation warehouses: " . $e->getMessage(),
                500,
                'MapShipstationWarehouses'
            );
        }
    }


    public function unmapWarehouses(Request $request){

        $request->validate(
            [
                'ids' => 'required|array',
                'ids.*' => 'required|exists:shipping_provider_warehouse_mappings,id'
            ],
        );

        try{
            $this->repository->deleteWarehouseMappings($request->input('ids'));
            return $this->response
                ->addData(
                    ShipstationWarehouseMappingResource::collection(
                        $this->repository->getWarehouseMappings()
                    )
                )
                ->success(message: 'Shipstation warehouses unmapped successfully.');
        }catch (\Exception $e){
            return $this->response->addError(
                "There was a problem unmapping Shipstation warehouses: " . $e->getMessage(),
                500,
                'UnmapShipstationWarehouses'
            );
        }
    }


    public function download(){

        $shipStationIntegrationInstance = IntegrationInstance::with([])->shipstation()->first();
        if (! $shipStationIntegrationInstance) {
            return $this->response->addError(__('messages.integration_instance.not_integrated', ['integration' => 'ShipStation']), 'ShipStation'.Response::CODE_NOT_INTEGRATED, 'shipstation');
        }

        try{
            $this->manager->downloadWarehouses($shipStationIntegrationInstance);
            return $this->response->success(message: 'Shipstation warehouses downloaded successfully.');
        }catch (ApiException $e){
            return $this->response->addError(
                "There was a problem downloading Shipstation warehouses: " . $e->getMessage(),
                500,
                'DownloadShipstationWarehouses'
            );
        }
    }

}