<?php

namespace App\Http\Controllers\ShipStation;

use App\Http\Controllers\IntegrationInstanceController;
use App\Http\Resources\ShippingMethodMappingShippingProviderResource;
use App\Integrations\ShipStation;
use App\Jobs\ShipStation\GetShippingServices;
use App\Jobs\ShipStation\SeedShippingServiceMapping;
use App\Jobs\ShipStation\SubmitOrders;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Models\ShipStation\ShipstationOrder;
use App\Response;
use App\SDKs\ShipStation\Model\XML\Orders;
use App\SDKs\ShipStation\Model\XML\ShipNotice;
use App\SDKs\ShipStation\ShipStationException;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Log;

/**
 * Class OrderController.
 */
class OrderController extends IntegrationInstanceController
{
    /**
     * OrderController constructor.
     */
    public function __construct()
    {
        parent::__construct();
    }

    public function submitOrders(IntegrationInstance $integrationInstance): Response
    {
        SubmitOrders::dispatch($integrationInstance);

        return $this->response->setMessage('Added to the Queue, it will be process shortly');
    }

    /**
     * Update ShipStation order from Sales order fulfillment.
     *
     *
     * @throws \App\SDKs\ShipStation\ShipStationException
     */
    public function updateOrderFromFulfillment(SalesOrderFulfillment $salesOrderFulfillment): Response
    {
        $shipStationIntegrationInstance = IntegrationInstance::with([])->shipstation()->first();
        if (! $shipStationIntegrationInstance) {
            return $this->response->addError(__('messages.integration_instance.not_integrated', ['integration' => 'ShipStation']), 'ShipStation'.Response::CODE_NOT_INTEGRATED, 'shipstation');
        }

        $shipStation = new ShipStation($shipStationIntegrationInstance);

        try {
            $shipstationOrder = $shipStation->submitOrder($salesOrderFulfillment->toShipStationOrder())->getResultAttribute()->toArray();
        } catch (ShipStationException $shipStationException) {
            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                $shipStationIntegrationInstance->unauthorizedConnection();
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError($shipStationException->getMessage(), Response::CODE_INTEGRATION_API_ERROR, 'id');
        }
        ShipstationOrder::with([])->updateOrCreate(['orderKey' => $shipstationOrder['orderKey']],
            ['json_data' => $shipstationOrder, 'sku_fulfillment_id' => $salesOrderFulfillment->id]);

        return $this->response->setMessage(__('messages.success.update', ['resource' => 'ShipStation Order', 'id' => $shipstationOrder['orderNumber']]));
    }

    public function downloadShippingServices(Request $request, IntegrationInstance $integrationInstance): Response
    {
        if ($integrationInstance->integration->name !== Integration::NAME_SHIPSTATION) {
            return $this->response->addError(
                'This is only available for Shipstation Integrations.',
                Response::CODE_UNACCEPTABLE,
                'DownloadShippingServices'
            );
        }

        try {
            dispatch_sync(new GetShippingServices($integrationInstance));
        } catch (ShipStationException $shipStationException) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)->addError($shipStationException->getMessage(), Response::CODE_INTEGRATION_API_ERROR, 'id');
        }

        return $this->seedShippingServices($request, $integrationInstance);
    }

    /**
     * @return Response|AnonymousResourceCollection
     */
    public function seedShippingServices(Request $request, IntegrationInstance $integrationInstance)
    {
        dispatch_sync(( new SeedShippingServiceMapping($integrationInstance) )->onConnection('sync'));

        $shippingProviderMappings = ShippingMethodMappingsSkuToShippingProviderMethod::with(['shippingMethod'])->where('shipping_provider_id', $integrationInstance->id)->get();

        return ShippingMethodMappingShippingProviderResource::collectionWithTableSpecifications($shippingProviderMappings, null);
    }

    /**
     * Export orders to ShipStation by way "Custom Store".
     *
     *
     * @return \Illuminate\Http\Response
     */
    public function shipStationExport(Request $request)
    {
        $request->validate(
            [
                'start_date' => 'required|date:m/d/Y H:m',
                'end_date' => 'required|date:m/d/Y H:m',
            ]
        );

        /* from parameters request to Carbon object */
        $start_date = Carbon::createFromFormat('m/d/Y H:m', $request->get('start_date'));
        $end_date = Carbon::createFromFormat('m/d/Y H:m', $request->get('end_date'));

        /**
         * Get Sales Orders between dates.
         */
        //    dd(Carbon::now()->format('m/d/Y H:m'));
        $sales_orders = SalesOrder::with('salesChannel', 'shippingAddress', 'billingAddress', 'salesOrderLines', 'customer')
            ->whereBetween('order_date', [$start_date, $end_date])
            ->paginate(10);

        /* new object from ShipStation xml orders */
        $xml_orders = new Orders(['Orders' => []]);
        foreach ($sales_orders as $sales_order) {
            $xml_order = $sales_order->toShipStationXmlModel();
            if (! $xml_order->validate()) {
                /**
                 * Currently just log errors.
                 */
                //Log::warning('Sales Order ('.$sales_order->id.'): '.json_encode($xml_order->errors));
                continue;
            }
            /* add valid order to shipStation orders*/
            $xml_orders->Orders[] = $xml_order;
        }

        /* return converted xml */

        return response()
            ->make($xml_orders->toXML()->saveXML())
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Received shipped orders from ShipStation.
     */
    public function shipStationShipNotify(Request $request): JsonResponse
    {
        $xml_content_to_array = json_decode(json_encode(simplexml_load_string($request->getContent())), true);

        $shipNotice = new ShipNotice($xml_content_to_array ?: []);

        // TODO: here we can set order as shipped

        return response()->json(['Received.']);
    }
}
